package carmel.value;

import carmel.interpreter.Heap;
import carmel.interpreter.VerificationException;
import carmel.tree.*;
import carmel.type.*;
import javax.swing.event.EventListenerList;

public class ClassValue extends NonNullReferenceValue {

    public boolean isInit = false;

    protected Value[] fieldValues;
    protected boolean[] fieldInit;
    protected int sizeInBytes;
    protected TreeClass parentClass;

    protected EventListenerList listeners = new EventListenerList();

    public ClassValue(Heap heap, TreeClass type) {
        super(type);

        parentClass = type;

        int fieldIDCount = parentClass.getFieldIDCount();
        fieldValues = new Value[fieldIDCount];
        fieldInit = new boolean[fieldIDCount];

        for (int i = fieldIDCount; --i >= 0;)
            fieldValues[i] = parentClass.getField(i).getType().getJCVMType().createDefaultValue();

        heapLocation = heap.allocate(this);
    }

    public Value getFieldValue(int fieldID) {
        return fieldValues[fieldID];
    }

    public void setFieldValue(int fieldID, Value value) throws TypeException, VerificationException {
        TreeField field = parentClass.getField(fieldID);

        field.getType().checkAssignableFrom(value.getType());

        if (field.isFinal()) {
            if (fieldInit[fieldID])
                throw new VerificationException("Attempt to set the value of a final field for the second time");
            else
                fieldInit[fieldID] = true;
        }

        fieldValues[fieldID] = value;

        fireFieldChanged(fieldID);
    }

    public short getSizeInBytes() {
        // +1 is for field count, stored in a byte
        return (short) (1 + parentClass.getFieldsSizeInBytes());
    }

    public TreeClass getParentClass() { return parentClass; }

    public void addClassValueListener(ClassValueListener l) {
        listeners.add(ClassValueListener.class, l);
    }

    public void removeClassValueListener(ClassValueListener l) {
        listeners.remove(ClassValueListener.class, l);
    }

    protected void fireFieldChanged(int index) {
        ClassValueEvent e = new ClassValueEvent(this, index);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == ClassValueListener.class)
                ((ClassValueListener) list[i + 1]).fieldChanged(e);
    }
}