package carmel.value;

import carmel.interpreter.Heap;
import carmel.interpreter.StackEntryList;
import carmel.interpreter.StackEntryListListener;
import carmel.type.*;
import java.util.*;
import javax.swing.event.EventListenerList;

public class ArrayValue extends NonNullReferenceValue implements StackEntryList {

    protected Value[] values;
    protected ComponentType componentType;
    protected short length;

    protected EventListenerList listeners = new EventListenerList();

    protected ArrayValue(short length, Heap heap, ArrayType type) {
        super(type);

        this.length = length;
        values = new Value[length];
        componentType = type.getComponentType();

        heapLocation = heap.allocate(this);
    }

    public ArrayValue(Heap heap, ArrayType type, short length) throws NegativeArraySizeException {
        this(length, heap, type);

        JCVMOperandType elementJCVMType = componentType.getJCVMType();

        while (--length >= 0)
            values[length] = elementJCVMType.createDefaultValue();
    }

    public ArrayValue(Heap heap, ArrayType type, List values) {
        this((short) values.size(), heap, type);

        int index = 0;

        for (Iterator i = values.iterator(); i.hasNext();)
            this.values[index++] = (Value) i.next();
    }

    public short getLength() { return length; }

    public short getSizeInBytes() {
        // +2 is for element count, stored in a short
        return (short) (length * componentType.getSizeInBytes() + 2);
    }

    public Value getValueAt(int index) throws ArrayIndexOutOfBoundsException {
        return values[index];
    }

    public void setValueAt(int index, Value value) throws ArrayIndexOutOfBoundsException, TypeException {
        componentType.checkAssignableFrom(value.getType());

        values[index] = value;

        fireElementChanged(index);
    }

    public ComponentType getComponentType() { return componentType; }

    public void addStackEntryListListener(StackEntryListListener l) {
        listeners.add(ArrayValueListener.class, l);
    }

    public void removeStackEntryListListener(StackEntryListListener l) {
        listeners.remove(ArrayValueListener.class, l);
    }

    protected void fireElementChanged(int index) {
        ArrayValueEvent e = new ArrayValueEvent(this, index);
        Object[] list = listeners.getListenerList();

        for (int i = list.length -2; i >= 0; i -= 2)
            if (list[i] == ArrayValueListener.class)
                ((ArrayValueListener) list[i + 1]).elementChanged(e);
    }

    public int getEntryListSize() { return length; }

    public StackEntry getEntryNoVerification(int index) {
        return values[index];
    }

    public int hashCode() {
        return componentType.hashCode() + Arrays.asList(values).hashCode();
    }

    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ArrayValue)) return false;
        ArrayValue v = (ArrayValue) o;
        return componentType == v.componentType && Arrays.equals(values, v.values);
    }
}