package carmel.type;

import carmel.value.*;

public class JCVMShortType extends JCVMNumericType {

    public static final JCVMShortType TYPE = new JCVMShortType();

    private JCVMShortType() {}

    public Value createDefaultValue() { return new ShortValue(); }

    public NumericValue to(int value) {
        return new ShortValue((short) value);
    }

    public NumericValue createValue(int value) throws IllegalArgumentException {
        if ((value < Short.MIN_VALUE) || (value > Short.MAX_VALUE))
            throw new IllegalArgumentException("Value out of range for short");

        return to(value);
    }

    public String getName() { return "s"; }

    public NumericValue applyUnaryNumOp(int operator, NumericValue value, JCVMNumericType resultType) throws TypeException {
        checkType(value);
        short v = ((ShortValue) value).getValue();

        switch (operator) {
            case NEG:
                return new ShortValue((short) -v);
            case TO:
                return resultType.to(v);
            case TOB:
                return resultType.to((byte) v);
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }
    }

    public NumericValue applyBinaryNumOp(int operator, NumericValue value1, NumericValue value2) throws TypeException {
        checkType(value1);
        checkType(value2);

        short v1 = ((ShortValue) value1).getValue();
        short v2 = ((ShortValue) value2).getValue();
        int result;

        switch (operator) {
            case ADD:
                result = v1 + v2;
                break;
            case SUB:
                result = v1 - v2;
                break;
            case MUL:
                result = v1 * v2;
                break;
            case DIV:
                result = v1 / v2;
                break;
            case REM:
                result = v1 % v2;
                break;
            case AND:
                result = v1 & v2;
                break;
            case OR:
                result = v1 | v2;
                break;
            case XOR:
                result = v1 ^ v2;
                break;
            case SHL:
                result = v1 << v2;
                break;
            case SHR:
                result = v1 >> v2;
                break;
            case USHR:
                result = v1 >>> v2;
                break;
            case CMP:
                result = v1 == v2 ? 0 : (v1 > v2 ? 1 : -1);
                break;
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }

        return new ShortValue((short) result);
    }
}