package carmel.type;

import carmel.value.*;

public class JCVMIntType extends JCVMNumericType {

    public static final JCVMIntType TYPE = new JCVMIntType();

    private JCVMIntType() {}

    public Value createDefaultValue() { return new IntValue(); }

    public NumericValue to(int value) {
        return new IntValue(value);
    }

    public NumericValue createValue(int value) throws IllegalArgumentException {
        return to(value);
    }

    public boolean isDoubleWord() { return true; }

    public String getName() { return "i"; }

    public NumericValue applyUnaryNumOp(int operator, NumericValue value, JCVMNumericType resultType) throws TypeException {
        checkType(value);
        int v = ((IntValue) value).getValue();

        switch (operator) {
            case NEG:
                return new IntValue(-v);
            case TO:
                return resultType.to(v);
            case TOB:
                return resultType.to((byte) v);
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }
    }

    public NumericValue applyBinaryNumOp(int operator, NumericValue value1, NumericValue value2) throws TypeException {
        checkType(value1);
        checkType(value2);

        int v1 = ((IntValue) value1).getValue();
        int v2 = ((IntValue) value2).getValue();
        int result;

        switch (operator) {
            case ADD:
                result = v1 + v2;
                break;
            case SUB:
                result = v1 - v2;
                break;
            case MUL:
                result = v1 * v2;
                break;
            case DIV:
                result = v1 / v2;
                break;
            case REM:
                result = v1 % v2;
                break;
            case AND:
                result = v1 & v2;
                break;
            case OR:
                result = v1 | v2;
                break;
            case XOR:
                result = v1 ^ v2;
                break;
            case SHL:
                result = v1 << v2;
                break;
            case SHR:
                result = v1 >> v2;
                break;
            case USHR:
                result = v1 >>> v2;
                break;
            case CMP:
                result = v1 == v2 ? 0 : (v1 > v2 ? 1 : -1);
                break;
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }

        return new IntValue(result);
    }
}