package carmel.type;

import carmel.value.*;

public class JCVMByteType extends JCVMNumericType {

    public static final JCVMByteType TYPE = new JCVMByteType();

    private JCVMByteType() {}

    public Value createDefaultValue() { return new ByteValue(); }

    public NumericValue to(int value) {
        return new ByteValue((byte) value);
    }

    public NumericValue createValue(int value) throws IllegalArgumentException {
        if ((value < Byte.MIN_VALUE) || (value > Byte.MAX_VALUE))
            throw new IllegalArgumentException("Value out of range for byte");

        return to(value);
    }

    public String getName() { return "b"; }

    public NumericValue applyUnaryNumOp(int operator, NumericValue value, JCVMNumericType resultType) throws TypeException {
        checkType(value);
        byte v = ((ByteValue) value).getValue();

        switch (operator) {
            case NEG:
                return new ByteValue((byte) -v);
            case TO:
            case TOB:
                return resultType.to(v);
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }
    }

    public NumericValue applyBinaryNumOp(int operator, NumericValue value1, NumericValue value2) throws TypeException {
        checkType(value1);
        checkType(value2);

        byte v1 = ((ByteValue) value1).getValue();
        byte v2 = ((ByteValue) value2).getValue();
        int result;

        switch (operator) {
            case ADD:
                result = v1 + v2;
                break;
            case SUB:
                result = v1 - v2;
                break;
            case MUL:
                result = v1 * v2;
                break;
            case DIV:
                result = v1 / v2;
                break;
            case REM:
                result = v1 % v2;
                break;
            case AND:
                result = v1 & v2;
                break;
            case OR:
                result = v1 | v2;
                break;
            case XOR:
                result = v1 ^ v2;
                break;
            case SHL:
                result = v1 << v2;
                break;
            case SHR:
                result = v1 >> v2;
                break;
            case USHR:
                result = v1 >>> v2;
                break;
            case CMP:
                result = v1 == v2 ? 0 : (v1 > v2 ? 1 : -1);
                break;
            default:
                throw new InternalError("Unknown binary numeric operator: " + operator);
        }

        return new ByteValue((byte) result);
    }
}