package carmel.tree;

import carmel.interpreter.Heap;
import carmel.parser.Token;
import carmel.type.*;
import carmel.value.ClassValue;
import java.util.*;

public class TreeClass extends TreeClassOrInterface {

    TreeClass superClass = null;
    Map constructors, fields;

    protected ClassReference superClassReference;
    protected List unmappedConstructors;

    protected short fieldsSizeInBytes;
    protected int fieldIDFrom;
    protected int fieldIDTo = -1;
    protected TreeField[] fieldArray;

    public TreeClass(Token nameToken, int modifiers, ClassReference superClass, List interfaces, Map staticFields, Map fields, List constructors, List methods) {
        super(nameToken, modifiers, interfaces, staticFields, methods);
        this.fields = fields;
        superClassReference = superClass;
        unmappedConstructors = constructors;
    }

    public TreeClass getSuperClass() { return superClass; }

    public Collection getDeclaredFields() { return fields.values(); }

    public TreeStaticField getStaticField(String name) throws NoSuchFieldException {
        try {
            return getDeclaredStaticField(name);
        }
        catch (NoSuchFieldException e) {
            if (superClass != null) {
                try {
                    return superClass.getStaticField(name);
                }
                catch (NoSuchFieldException e2) {}
            }

            return getStaticFieldFromInterfaces(name);
        }
    }

    public TreeField getDeclaredField(String name) throws NoSuchFieldException {
        TreeField field = (TreeField) fields.get(name);

        if (field != null) return field;

        throw new NoSuchFieldException("Field " + name + " not found");
    }

    public TreeField getField(int fieldID) {
        if (fieldID < 0) throw new IndexOutOfBoundsException("negative field ID");
        return (fieldID < fieldIDFrom) ? superClass.getField(fieldID) : fieldArray[fieldID - fieldIDFrom];
    }

    public TreeField getField(String name) throws NoSuchFieldException {
        try {
            return getDeclaredField(name);
        }
        catch (NoSuchFieldException e) {
            if (superClass != null) return superClass.getDeclaredField(name);
            throw e;
        }
    }

    public TreeMethod getMethod(MethodID methodID) throws NoSuchMethodException {
        try {
            return getDeclaredMethod(methodID);
        }
        catch (NoSuchMethodException e) {
            if (superClass != null) {
                TreeMethod method = superClass.getMethod(methodID);

                if (method.isPrivate())
                    throw new NoSuchMethodException("Method " + methodID + " is declared private in class " + method.getParentClass().getName());

                return method;
            }

            throw e;
        }
    }

    public Collection getConstructors() { return constructors.values(); }

    public TreeConstructor getConstructor(List parameterTypes) throws NoSuchMethodException {
        TreeConstructor constructor = (TreeConstructor) constructors.get(parameterTypes);

        if (constructor != null) return constructor;

        throw new NoSuchMethodException("Constructor " + getName() + new MethodID("", parameterTypes) + " not found");
    }

    public TreeConstructor getDefaultConstructor() throws NoSuchMethodException {
        return getConstructor(Collections.EMPTY_LIST);
    }

    public boolean isAssignableFrom(ResultType type) {
        return this == type || type == NullType.TYPE || ((type instanceof TreeClass) && isAssignableFrom(((TreeClass) type).superClass));
    }

    public ClassValue newInstance(Heap heap) {
        return new ClassValue(heap, this);
    }

    protected int assignFieldIDs() {
        if (fieldIDTo == -1) {
            fieldIDTo = superClass == null ? 0 : (fieldIDFrom = superClass.assignFieldIDs());

            Collection declaredFields = fields.values();
            fieldArray = new TreeField[declaredFields.size()];
            fieldsSizeInBytes = superClass == null ? 0 : superClass.fieldsSizeInBytes;
            int index = 0;

            for (Iterator i = declaredFields.iterator(); i.hasNext();) {
                TreeField field = (TreeField) i.next();

                field.fieldID = fieldIDTo++;
                fieldsSizeInBytes += field.getType().getSizeInBytes();

                fieldArray[index++] = field;
            }
        }

        return fieldIDTo;
    }

    public int getFieldIDCount() {
        return fieldIDTo;
    }

    public short getFieldsSizeInBytes() {
        return fieldsSizeInBytes;
    }

    public void visit(Visitor v) throws Exception {
        v.visit(this);
    }
}