package carmel.tree;

import carmel.interpreter.CarmelClassLoader;
import carmel.parser.*;
import carmel.type.*;
import carmel.value.Value;
import java.util.*;

public class LinkPass2Visitor extends DefaultVisitor {

    protected CarmelClassLoader classLoader;
    protected TreeClassOrInterface currentClass;
    protected int localVariableArraySize;

    public LinkPass2Visitor(CarmelClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    public void visitPackage(TreePackage p) throws ParseException {
        try {
            visitCollection(p.classes.values());
            visitCollection(p.interfaces.values());
        }
        catch (ParseException e) {
            e.setSource(p.source);
            throw e;
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        }
    }

    public void visit(TreeClass c) throws Exception {
        currentClass = c;

        if (c.isPublic() && c.superClass != null && c.superClass.hasDefaultAccess()) {
            for (TreeClass sc = c.superClass; sc != null; sc = sc.superClass) {
                for (Iterator i = sc.getDeclaredMethods().iterator(); i.hasNext();) {
                    TreeMethod m = (TreeMethod) i.next();

                    if (m.isPublic())
                        throw new ParseException("Cannot extend class " + sc.getName() + " with default access and public method " + m.getNameWithTypes() + " with a public class", c.nameToken);
                    if (m.isProtected())
                        throw new ParseException("Cannot extend class " + sc.getName() + " with default access and protected method " + m.getNameWithTypes() + " with a public class", c.nameToken);
                }
            }
        }

        checkClassCircularity(c, new HashSet());

        visitCollection(c.constructors.values());
        visitCollection(c.methods.values());

        c.assignFieldIDs();
    }

    public void visit(TreeInterface i) throws Exception {
        currentClass = i;

        if (i.isPublic()) {
            for (Iterator it = i.interfaces.iterator(); it.hasNext();) {
                TreeInterface i2 = (TreeInterface) it.next();

                if (i2.hasDefaultAccess())
                    throw new ParseException("Cannot extend interface " + i2.getName() + " with default access with a public interface", i.nameToken);
            }
        }

        checkInterfaceCircularity(i, new HashSet());
    }

    protected void visitTreeConstructorOrMethod(TreeConstructorOrMethod m) throws Exception {
        localVariableArraySize = m.localVariableArraySize;

        if (m.exceptions != null) {
            Iterator i = m.exceptions.iterator();
            Iterator j = m.exceptionReferences.iterator();

            while (i.hasNext()) {
                TreeClass c = (TreeClass) i.next();
                ClassReference r = (ClassReference) j.next();

                if (!classLoader.throwable.isAssignableFrom(c))
                    throw new ParseException("java.lang.Throwable expected, " + c.getName() + " found", r.firstToken);
            }

            m.exceptionReferences = null;
        }

        visit(m.instructionBlock);
    }

    public void visit(TreeConstructor c) throws Exception {
        visitTreeConstructorOrMethod(c);
    }

    protected void checkMethodOverride(TreeMethod method, TreeMethod overriddenMethod) throws ParseException {
        // private methods are not actually overridden
        if (overriddenMethod.isPrivate()) return;

        if (overriddenMethod.isPublic() && (method.isPrivate() || method.isProtected() || !method.isPublic()))
            throw new ParseException("Cannot override method with weaker access privileges, was public in " + overriddenMethod.getParentClass().getName(), method.nameToken);

        if (overriddenMethod.hasDefaultAccess() && !method.hasDefaultAccess())
            throw new ParseException("Cannot override method with default access in " + overriddenMethod.getParentClass().getName() + " with a different access privilege", method.nameToken);

        if (overriddenMethod.isProtected() && method.isPrivate())
            throw new ParseException("Cannot override method with weaker access privileges, was protected in " + overriddenMethod.getParentClass().getName(), method.nameToken);

        if (overriddenMethod.isFinal())
            throw new ParseException("Cannot override a final method", method.nameToken);

        if (method.getResultType() != overriddenMethod.getResultType())
            throw new ParseException("Cannot override method with a different return type, was " + overriddenMethod.getResultType().getName() + " in " + overriddenMethod.getParentClass().getName(), method.nameToken);
    }

    public void visit(TreeMethod m) throws Exception {
        visitTreeConstructorOrMethod(m);

        TreeMethod overriddenMethod = null;

        if (currentClass instanceof TreeClass) {
            TreeClass superClass = ((TreeClass) currentClass).getSuperClass();

            if (superClass != null) {
                try {
                    checkMethodOverride(m, ((TreeClass) currentClass).getSuperClass().getMethod(m.methodID));
                }
                catch (NoSuchMethodException e) {}
            }

            if (currentClass.isPublic() && (m.isPublic() || m.isProtected())) {
                for (Iterator i = m.getParameterTypes().iterator(); i.hasNext();) {
                    TreeClass c;

                    try {
                        c = (TreeClass) i.next();

                        if (c.hasDefaultAccess())
                            throw new ParseException("Public method " + m.getNameWithTypes() + " in " + (m.isPublic() ? "public" : "protected") + " class " + currentClass.getName() + " cannot have parameter " + c.getName() + " with default access", m.nameToken);
                    }
                    catch (ClassCastException e) {
                    }
                }
            }
        }

        Collection interfaces = currentClass.getInterfaces();

        if (interfaces != null) {
            for (Iterator i = interfaces.iterator(); i.hasNext();) {
                try {
                    checkMethodOverride(m, ((TreeInterface) i.next()).getMethod(m.methodID));
                }
                catch (NoSuchMethodException e) {}
            }
        }
    }

    public void visit(InstructionBlock i) throws Exception {
        Instruction instruction = i.firstInstruction;

        do {
            instruction.visit(this);
        }
        while ((instruction = instruction.next) != null);

        visitCollection(i.handlers);
    }

    public void visit(ExceptionHandler h) throws Exception {
        if ((h.catchType != null) && (!classLoader.throwable.isAssignableFrom(h.catchType)))
            throw new ParseException("java.lang.Throwable expected, " + h.catchType.getName() + " found", h.catchTypeReference.firstToken);

        h.catchTypeReference = null;
    }

    protected void visitStaticFieldInstruction(FieldInstruction i) throws Exception {
        checkMemberAccess(i.field, i.fieldReference.firstToken);
        i.fieldReference = null;
    }

    protected void visitFieldInstruction(FieldInstruction i) throws Exception {
        checkMemberAccess(i.field, i.fieldReference.firstToken);
        i.fieldReference = null;
    }

    protected void visitLocalVariableInstruction(LocalVariableInstruction i) throws Exception {
        if (i.index + (i.type.isDoubleWord() ? 1 : 0) >= localVariableArraySize)
            throw new ParseException("Local variable index invalid", i.token);
    }

    public void visit(InvokeConstructorInstruction i) throws Exception {
        try {
            i.constructor = i.constructorReference.parentClass.getConstructor(i.constructorReference.types);
            checkMemberAccess(i.constructor, i.constructorReference.firstToken);
        }
        catch (NoSuchMethodException e) {
            throw new ParseException(e.getMessage(), i.constructorReference.firstToken);
        }

        i.constructorReference = null;
    }

    public void visit(InvokeDefiniteMethodInstruction i) throws Exception {
        try {
            i.method = i.methodReference.parentClass.getMethod(i.methodReference.methodID);
            checkMemberAccess(i.method, i.methodReference.firstToken);
        }
        catch (NoSuchMethodException e) {
            throw new ParseException(e.getMessage(), i.methodReference.firstToken);
        }

        i.methodReference = null;
    }

    public void visit(InvokeVirtualInstruction i) throws Exception {
        // just check that at least one method really exists
        // the true method will be resolved at run time
        try {
            TreeMethod method = i.parentClass.getMethod(i.methodID);

            if (method.isStatic()) throw new ParseException("Static method not allowed here", i.methodReference.firstToken);
            checkMemberAccess(method, i.methodReference.firstToken);
            // todo: int methodID
        }
        catch (NoSuchMethodException e) {
            throw new ParseException(e.getMessage(), i.methodReference.firstToken);
        }

        i.methodReference = null;
    }

    public void visit(InvokeInterfaceInstruction i) throws Exception {
        // just check that at least one method really exists
        // the true method will be resolved at run time
        try {
            TreeMethod method = i.parentInterface.getMethod(i.methodID);

            if (method.isStatic()) throw new ParseException("Static method not allowed here", i.methodReference.firstToken);
            checkMemberAccess(method, i.methodReference.firstToken);
        }
        catch (NoSuchMethodException e) {
            throw new ParseException(e.getMessage(), i.methodReference.firstToken);
        }

        i.methodReference = null;
    }

    protected void checkClassCircularity(TreeClass c, Set superClasses) throws ParseException {
        if (!superClasses.add(c)) throw new ParseException("Class circularity error involving " + c.getName(), c.nameToken);
        if (c.superClass != null) checkClassCircularity(c.superClass, superClasses);
    }

    protected void checkInterfaceCircularity(TreeInterface i, HashSet superInterfaces) throws ParseException {
        if (!superInterfaces.add(i)) throw new ParseException("Class circularity error involving " + i.getName(), i.nameToken);
        for (Iterator it = i.interfaces.iterator(); it.hasNext();)
            checkInterfaceCircularity((TreeInterface) it.next(), (HashSet) superInterfaces.clone());
    }

    protected void checkMemberAccess(TreeClassMember m, Token token) throws ParseException {
        if (!m.isAccessibleFrom(currentClass))
            throw new ParseException("Class does not have the right to access " + m.getName(), token);
    }
}