package carmel.tree;

import carmel.parser.*;
import carmel.interpreter.CarmelClassLoader;
import carmel.type.*;
import carmel.value.Value;
import java.lang.reflect.Modifier;
import java.util.*;

public class LinkPass1Visitor extends DefaultInstructionVisitor implements Visitor {

    CarmelClassLoader classLoader;

    Map addressMap;
    int maxLocalVariableIndex;

    TreePackage currentPackage;
    TreeClassOrInterface currentClass;
    TreeConstructorOrMethod currentMethod;

    public LinkPass1Visitor(CarmelClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    public void visitPackage(TreePackage p) throws LexOrParseException {
        try {
            currentPackage = p;

            visit(p.imports);
            visitCollection(p.classes.values());
            visitCollection(p.interfaces.values());

            p.imports = null;
        }
        catch (LexOrParseException e) {
            e.setSource(p.source);
            throw e;
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        }
    }

    public void visit(ImportDeclaration i) throws Exception {
        Iterator it = i.packageImports.iterator();
        i.packageImports = new HashSet();

        while (it.hasNext()) {
            PackageReference r = (PackageReference) it.next();
            String filename = CarmelClassLoader.getPackageFilename(r.name);

            // check that the package is valid
            if (ClassLoader.getSystemResourceAsStream(filename) == null)
                throw new ParseException("File for package import " + r.name + " not found by appending " + filename + " to each path in CLASSPATH", r.firstToken);

            i.packageImports.add(r.name);
        }

        it = i.classImports.entrySet().iterator();
        i.classImports = new HashMap();

        while (it.hasNext()) {
            Map.Entry e = (Map.Entry) it.next();
            i.classImports.put(e.getKey(), resolveClassOrInterface((ClassReference) e.getValue()));
        }
    }

    protected void visitClassOrInterface(TreeClassOrInterface c) throws Exception {
        currentClass = c;

        c.parentPackage = currentPackage;

        if (c.interfaces != null) {
            Iterator i = c.interfaces.iterator();
            c.interfaces = new LinkedList();

            while (i.hasNext())
                c.interfaces.add(resolveInterface((ClassReference) i.next()));
        }

        visitCollection(c.staticFields.values());

        Iterator i = c.unmappedMethods.iterator();
        c.methods = new HashMap();

        while (i.hasNext()) {
            TreeMethod m = (TreeMethod) i.next();
            visit(m);
            if (c.methods.put(m.methodID, m) != null) {
                throw new ParseException("Duplicate definition of method " + m.getName(), m.nameToken);
            }
        }
    }

    public void visit(TreeClass c) throws Exception {
        if (c.superClassReference != null) c.superClass = resolveClass(c.superClassReference);

        visitClassOrInterface(c);

        // extending Object is caught by the class circularity check
        if (c.superClassReference == null) {
            try {
                TreeClass object = (TreeClass) classLoader.loadClass("java.lang.Object", 0);

                c.superClass = (c == object) ? null : object;
            }
            catch (ClassCastException e) {
                throw new ParseException("Implied superclass java.lang.Object must be a class", c.nameToken);
            }
            catch (ClassNotFoundException e) {
                throw new ParseException("Implied superclass java.lang.Object not found", c.nameToken);
            }
        }
        else {
            if (c.superClass.isFinal())
                throw new ParseException("Cannot extend a final class", c.superClassReference.firstToken);
        }

        visitCollection(c.fields.values());

        Iterator i = c.unmappedConstructors.iterator();
        c.constructors = new HashMap();

        while (i.hasNext()) {
            TreeConstructor co = (TreeConstructor) i.next();
            visit(co);
            if (c.constructors.put(co.parameterTypes, co) != null) {
                throw new ParseException("Duplicate definition of constructor", co.nameToken);
            }
        }
    }

    public void visit(TreeInterface i) throws Exception {
        visitClassOrInterface(i);
    }

    protected void visitClassMember(TreeClassMember m) {
        m.parentClass = currentClass;
    }

    protected void visitConstructorOrMethod(TreeConstructorOrMethod m) throws Exception {
        currentMethod = m;
        visitClassMember(m);

        m.parameterTypes = resolveTypeList(m.parameterTypes, new ArrayList(m.parameterTypes.size()));
        m.exceptions = resolveClassList(m.exceptionReferences);

        m.instructionBlock.visit(this);

        m.localVariableArraySize = maxLocalVariableIndex + 1;
    }

    public void visit(TreeConstructor c) throws Exception {
        maxLocalVariableIndex = c.parameterTypes.size() + 1;

        visitConstructorOrMethod(c);
    }

    public void visit(TreeMethod m) throws Exception {
        if (!currentClass.isAbstract() && m.isAbstract())
            throw new ParseException("Abstract method not allowed in non-abstract class", m.nameToken);

        if (currentClass instanceof TreeInterface && m.isNative())
            throw new ParseException("Native method not allowed in interface", m.nameToken);

        maxLocalVariableIndex = m.parameterTypes.size();

        if (!m.isStatic()) maxLocalVariableIndex++;

        m.resultType = resolveType(m.resultType);

        visitConstructorOrMethod(m);

        m.methodID = new MethodID(m.getMemberName(), m.getParameterTypes());
    }

    protected void visitAbstractField(TreeAbstractField f) throws Exception {
        visitClassMember(f);

        // void already checked in parser
        f.type = (Type) resolveType(f.type);
    }

    public void visit(TreeStaticField f) throws Exception {
        visitAbstractField(f);

        f.value.visit(this);
    }

    public void visit(TreeField f) throws Exception {
        visitAbstractField(f);
    }

    public void visit(Value v) throws Exception {
        // void already checked in parser
        v.type = (Type) resolveType(v.type);
    }

    public void visit(InstructionBlock i) throws Exception {
        addressMap = i.addressMap;
        i.addressMap = null;

        Instruction instruction = i.firstInstruction;

        while (instruction.next != null) {
            instruction.visit(this);
            instruction = instruction.next;
        }

        instruction.visit(this);

        if (!((instruction instanceof ReturnInstruction) || (instruction instanceof GotoInstruction) || (instruction instanceof ThrowInstruction)))
            throw new ParseException("Last instruction does not return a value, throw an exception or goto another address", instruction.token);

        visitCollection(i.handlers);
    }

    public void visit(ExceptionHandler h) throws Exception {
        h.catchType = resolveClass(h.catchTypeReference);
//        leave this for the second pass
//        h.catchTypeReference = null;

        h.fromIndex = resolveAddress(h.fromToken).blockIndex;
        h.toIndex = resolveAddress(h.toToken).blockIndex;
        h.targetInstruction = resolveAddress(h.targetToken);

        if (h.fromIndex > h.toIndex)
            throw new ParseException("Invalid exception handler address range", h.fromToken);

        if ((h.targetInstruction.blockIndex >= h.fromIndex) && (h.targetInstruction.blockIndex <= h.toIndex))
            throw new ParseException("Target address within catch range", h.targetToken);

        h.fromToken = h.toToken = h.targetToken = null;
    }

    protected void visitAddressInstruction(AddressInstruction i) throws Exception {
        i.instruction = resolveAddress(i.addressToken);
        i.addressToken = null;
    }

    public void visit(StoreInstruction i) throws Exception {
        if (i.index >= maxLocalVariableIndex)
            maxLocalVariableIndex = i.type.isDoubleWord() ? i.index + 1 : i.index;
    }

    public void visit(InvokeConstructorInstruction i) throws Exception {
        visit(i.constructorReference);
    }

    public void visit(InvokeDefiniteMethodInstruction i) throws Exception {
        visit(i.methodReference);

        if (!(i.methodReference.parentClass instanceof TreeClass))
            throw new ParseException("Interface method not allowed here", i.methodReference.firstToken);
    }

    public void visit(InvokeVirtualInstruction i) throws Exception {
        i.methodReference.visit(this);

        if (!(i.methodReference.parentClass instanceof TreeClass))
            throw new ParseException("Interface method not allowed here", i.methodReference.firstToken);

        i.parentClass = (TreeClass) i.methodReference.parentClass;
        i.methodID = i.methodReference.methodID;
    }

    public void visit(InvokeInterfaceInstruction i) throws Exception {
        i.methodReference.visit(this);

        if (!(i.methodReference.parentClass instanceof TreeInterface))
            throw new ParseException("Class method not allowed here", i.methodReference.firstToken);

        i.parentInterface = (TreeInterface) i.methodReference.parentClass;
        i.methodID = i.methodReference.methodID;
    }

    protected void visitStaticFieldInstruction(StaticFieldInstruction i) throws Exception {
        FieldReference r = i.fieldReference;

        TreeClassOrInterface c = r.classReference == null ? currentClass : resolveClassOrInterface(r.classReference);

        try {
            i.field = c.getDeclaredStaticField(r.name);
        }
        catch (NoSuchFieldException e) {
            throw new ParseException(e.getMessage(), r.firstToken);
        }
    }

    protected void visitFieldInstruction(FieldInstruction i) throws Exception {
        FieldReference r = i.fieldReference;

        try {
            TreeClass c = r.classReference == null ? (TreeClass) currentClass : resolveClass(r.classReference);
            i.field = c.getDeclaredField(r.name);
        }
        catch (ClassCastException e) {
            throw new ParseException("Interface does not have non-static fields", r.firstToken);
        }
        catch (NoSuchFieldException e) {
            throw new ParseException(e.getMessage(), r.firstToken);
        }
    }

    protected void visitTypeInstruction(TypeInstruction i) throws Exception {
        i.type = (ReferenceType) resolveType(i.type);
    }

    public void visit(NewArrayInstruction i) throws Exception {
        resolveType(i.type);
    }

    public void visit(NewClassInstruction i) throws Exception {
        i.classType = resolveClass(i.classReference);

        if (i.classType.isAbstract())
            throw new ParseException("Cannot instantiate an abstract class", i.classReference.firstToken);

        i.classReference = null;
    }

    public void visit(LookupSwitchInstruction i) throws Exception {
        super.visit(i);

        Iterator it = i.switches.entrySet().iterator();
        // not strictly required, may be marginally faster
        i.switches = new HashMap();

        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry) it.next();
            i.switches.put(entry.getKey(), resolveAddress((Token) entry.getValue()));
        }
    }

    public void visit(TableSwitchInstruction i) throws Exception {
        super.visit(i);

        Iterator it = i.addresses.iterator();
        i.addresses = new ArrayList(i.addresses.size());

        while (it.hasNext())
            i.addresses.add(resolveAddress((Token) it.next()));
    }

    public void visit(ReturnInstruction i) throws Exception {
        if (currentMethod instanceof TreeConstructor) {
            if (i.type != null) throw new ParseException("Constructor can only return void", i.token);
        }
        else {
            TreeMethod method = (TreeMethod) currentMethod;
            if (i.type == null) {
                if (method.resultType != VoidType.TYPE)
                    throw new ParseException("Method is declared to return " + method.resultType.getName() + ", but returns void", i.token);
            }
            else {
                if (method.resultType == VoidType.TYPE)
                    throw new ParseException("Method is declared to return void, but returns " + i.type.getName(), i.token);

                JCVMType type = ((Type) method.resultType).getJCVMType();
                if (type == JCVMByteType.TYPE) type = JCVMShortType.TYPE;

                if (!type.equals(i.type))
                    throw new ParseException("Method is declared to return " + method.resultType.getName() + ", but returns " + i.type.getName(), i.token);
            }
        }
    }

    public void visit(MethodReference r) throws Exception {
        r.parentClass = r.classReference == null ? currentClass : resolveClassOrInterface(r.classReference);

        r.methodID.types = resolveTypeList(r.methodID.types, new ArrayList(r.methodID.types.size()));
    }

    public void visit(ConstructorReference r) throws Exception {
        try {
            r.parentClass = r.classReference == null ? (TreeClass) currentClass : resolveClass(r.classReference);
        }
        catch (ClassCastException e) {
            throw new ParseException("Cannot construct an interface", r.firstToken);
        }

        r.types = resolveTypeList(r.types, new ArrayList(r.types.size()));
    }

    protected void visitCollection(Collection c) throws Exception {
        for (Iterator i = c.iterator(); i.hasNext();)
            ((Visitee) i.next()).visit(this);
    }

    protected TreeClassOrInterface loadClass(String name, Token token) throws LexOrParseException, ClassNotFoundException {
        TreeClassOrInterface c = classLoader.loadClass(name, 0);
        checkClassAccess(c, token);
        return c;
    }

    protected TreeInterface resolveInterface(ClassReference reference) throws Exception {
        try {
            return (TreeInterface) resolveType(reference);
        }
        catch (ClassCastException e) {
            throw new ParseException("Interface expected", reference.firstToken);
        }
    }

    protected TreeClass resolveClass(ClassReference reference) throws Exception {
        try {
            return (TreeClass) resolveType(reference);
        }
        catch (ClassCastException e) {
            throw new ParseException("Class expected", reference.firstToken);
        }
    }

    protected TreeClassOrInterface resolveClassOrInterface(ClassReference reference) throws Exception {
        try {
            return (TreeClassOrInterface) resolveType(reference);
        }
        catch (ClassCastException e) {
            throw new ParseException("Class or interface expected", reference.firstToken);
        }
    }

    protected ResultType resolveType(ResultType reference) throws LexOrParseException {
        if (reference instanceof ClassReference) {
            ClassReference r = (ClassReference) reference;
            TreeClassOrInterface c;

            if (r.packageReference == null || r.packageReference.name.equals(currentPackage.getName())) {
                // JLS2 6.5.5.1

                // check directly in the tree, as the class loader has not yet cached it
                // doing otherwise will get us into an infinite loop

                // there is no need to check accessibility, we are in the same package
                c = currentPackage.getClassOrInterface(r.name);
                if (c != null) return c;

                c = (TreeClassOrInterface) currentPackage.imports.classImports.get(r.name);
                if (c != null) return c;

                if (r.packageReference != null)
                    throw new ParseException("Class " + r.getName() + " not found", r.firstToken);
            }
            else {
                // JLS2 6.5.5.2
                try {
                    return loadClass(r.getName(), r.firstToken);
                }
                catch (ClassNotFoundException e) {
                    throw new ParseException(e.getMessage(), r.firstToken);
                }
            }

            // JLS2 6.5.5.1 continued...

            // how the unnamed package is handled is not specified in JLS
            // here we treat it as if it is implicitly imported by every package
            try {
                c = loadClass(r.name, r.firstToken);
            }
            catch (ClassNotFoundException e) {}

            String suffix = "." + r.name;
            TreeClassOrInterface temp;

            try {
                temp = loadClass("java.lang" + suffix, r.firstToken);

                if (c == null)
                    c = temp;
                else
                    throw new ParseException(r.name + " exists in more than one package import", r.firstToken);
            }
            catch (ClassNotFoundException e) {}

            for (Iterator i = currentPackage.imports.packageImports.iterator(); i.hasNext();) {
                try {
                    temp = loadClass(((String) i.next()) + suffix, r.firstToken);

                    if (c == null)
                        c = temp;
                    else
                        throw new ParseException(r.name + " exists in more than one package import", r.firstToken);
                }
                catch (ClassNotFoundException e) {}
            }

            if (c != null) return c;

            throw new ParseException("Class " + r.name + " not found", r.firstToken);
        }
        else if (reference instanceof ArrayType) {
            ((ArrayType) reference).componentType = (ComponentType) resolveType(((ArrayType) reference).componentType);
        }

        return reference;
    }

    protected List resolveClassList(List list) throws Exception {
        return list == null ? null : resolveClassList(list, new LinkedList());
    }

    protected List resolveClassList(List oldList, List newList) throws Exception {
        if (oldList == null) return null;

        for (Iterator i = oldList.iterator(); i.hasNext();) {
            newList.add(resolveClass((ClassReference) i.next()));
        }

        return newList;
    }

    protected List resolveTypeList(List oldList, List newList) throws Exception {
        if (oldList == null) return null;

        for (Iterator i = oldList.iterator(); i.hasNext();) {
            newList.add(resolveType((ResultType) i.next()));
        }

        return newList;
    }

    protected Instruction resolveAddress(Token token) throws ParseException {
        Instruction instruction = (Instruction) addressMap.get(token.image);

        if (instruction == null) throw new ParseException("Address " + token.image + " not found", token);

        return instruction;
    }

    protected void checkClassAccess(TreeClassOrInterface c, Token token) throws ParseException {
        if (!c.isAccessibleFrom(currentClass))
            throw new ParseException("Class does not have the right to access " + c.getName(), token);
    }
}