package carmel.interpreter;

import carmel.parser.LexOrParseException;
import carmel.tree.*;
import carmel.type.TypeException;
import carmel.value.*;
import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import javax.swing.event.EventListenerList;

public class VirtualMachine {

    public final ClassValue
        nullPointerException, arithmeticException, classCastException,
        negativeArraySizeException, arrayIndexOutOfBoundsException,
        arrayStoreException, securityException;

    protected Heap heap = new Heap();
    protected CallStack callStack = new CallStack();
    protected CarmelClassLoader classLoader;

    protected EventListenerList listeners = new EventListenerList();

    protected StepIntoListener stepIntoListener = new StepIntoListener();
    protected StepOverListener stepOverListener = new StepOverListener();
    protected StepOutListener stepOutListener = new StepOutListener();

    protected boolean wait = false;
    protected boolean halt = false;

    public VirtualMachine() throws ClassNotFoundException, NoSuchMethodException, LexOrParseException, CarmelException, VerificationException, VMException {
        classLoader = new CarmelClassLoader(heap);

        nullPointerException =              createInstance(loadClass("java.lang.NullPointerException"));
        arithmeticException =               createInstance(loadClass("java.lang.ArithmeticException"));
        negativeArraySizeException =        createInstance(loadClass("java.lang.NegativeArraySizeException"));
        classCastException =                createInstance(loadClass("java.lang.ClassCastException"));
        arrayIndexOutOfBoundsException =    createInstance(loadClass("java.lang.ArrayIndexOutOfBoundsException"));
        arrayStoreException =               createInstance(loadClass("java.lang.ArrayStoreException"));
        securityException =                 createInstance(loadClass("java.lang.SecurityException"));

        callStack.addCallStackListener(new CallStackToPCListener());
    }

    private TreeClass loadClass(String name) throws ClassNotFoundException, LexOrParseException {
        try {
            return ((TreeClass) classLoader.loadClass(name));
        }
        catch (ClassCastException e) {
            throw new ClassNotFoundException(name + " must not be an interface");
        }
    }

    protected ClassValue createInstance(TreeClass c) throws NoSuchMethodException, CarmelException, VerificationException, VMException {
        ClassValue value = new ClassValue(heap, c);

        try {
            new StackFrame(this, c.getDefaultConstructor(), Collections.EMPTY_LIST, value).interpret();
        }
        catch (HaltException e) {
            throw new VMException("Unexpected halt exception", e);
        }

        return value;
    }

    public void invokeInstanceMethod(TreeMethod method, ClassValue value) throws IllegalArgumentException {
        if (method.isStatic())
            throw new IllegalArgumentException("Method must not be static");

        if (!(method.getParentClass() instanceof TreeClass))
            throw new IllegalArgumentException("Cannot invoke an interface method");

        if (!method.getParameterTypes().isEmpty())
            throw new IllegalArgumentException("Method must not take any parameters");

        callStack.clear();

        try {
            interpretFrame(new StackFrame(this, method, Collections.EMPTY_LIST, value));
        }
        catch (TypeException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    public void invokeStaticMethod(TreeMethod method) throws IllegalArgumentException {
        if (!method.isStatic())
            throw new IllegalArgumentException("Method must be static");

        if (!(method.getParentClass() instanceof TreeClass))
            throw new IllegalArgumentException("Cannot invoke an interface method");

        if (!method.getParameterTypes().isEmpty())
            throw new IllegalArgumentException("Method must not take any parameters");

        callStack.clear();

        interpretFrame(new StackFrame(this, method, Collections.EMPTY_LIST));
    }

    public void invokeConstructor(TreeConstructor c) throws IllegalArgumentException {
        if (!c.getParameterTypes().isEmpty())
            throw new IllegalArgumentException("Constructor must not take any parameters");

        callStack.clear();

        try {
            interpretFrame(new StackFrame(this, c, Collections.EMPTY_LIST, new ClassValue(heap, (TreeClass) c.getParentClass())));
        }
        catch (TypeException e) {
            throw new Error("Unexpected TypeException");
        }
    }

    protected void interpretFrame(final StackFrame frame) {
        halt = false;
        wait = true;

        // thread will wait just before first instruction
        new Thread(
            new Runnable() {
                public void run() {
                    try {
                        fireMethodReturned(frame.interpret());
                    }
                    catch (HaltException e) {
                        callStack.clear();
                        fireVMHalted();
                    }
                    catch (CarmelException e) {
                        callStack.setStack(e.getCallStack());
                        fireMethodReturned(e);
                    }
                    catch (VerificationException e) {
                        fireVMException(e);
                    }
                    catch (VMException e) {
                        fireVMException((Exception) e.getCause());
                    }
                }
            }
        ).start();
    }

    public Heap getHeap() { return heap; }
    public CarmelClassLoader getClassLoader() { return classLoader; }
    public CallStack getCallStack() { return callStack; }

    public void addPCListener(PCListener l) {
        listeners.add(PCListener.class, l);
    }

    public void removePCListener(PCListener l) {
        listeners.remove(PCListener.class, l);
    }

    public void addVMListener(VirtualMachineListener l) {
        listeners.add(VirtualMachineListener.class, l);
    }

    public void removeVMListener(VirtualMachineListener l) {
        listeners.remove(VirtualMachineListener.class, l);
    }

    protected void resume() {
        if (wait) {
            wait = false;
            notify();
        }
    }

    public synchronized void run() {
        resume();
    }

    public synchronized void halt() {
        halt = true;
        notify();
    }

    public synchronized void stepInto() {
        resume();
        addPCListener(stepIntoListener);
    }

    public synchronized void stepOver() {
        resume();
        callStack.peek().addStackFrameListener(stepOverListener);
    }

    public synchronized void stepOut() {
        resume();
        callStack.peek().addStackFrameListener(stepOutListener);
    }

    protected synchronized void waitForNextInstruction() throws HaltException {
        if (halt) throw new HaltException();

        while (wait) {
            try {
                wait();

                if (halt) throw new HaltException();
            }
            catch (InterruptedException e) {}
        }
    }

    protected void fireMethodReturned(Value v) {
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == VirtualMachineListener.class)
                ((VirtualMachineListener) list[i + 1]).methodReturned(v);
    }

    protected void fireMethodReturned(CarmelException e) {
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == VirtualMachineListener.class)
                ((VirtualMachineListener) list[i + 1]).methodReturned(e);
    }

    protected void fireVMException(VerificationException e) {
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == VirtualMachineListener.class)
                ((VirtualMachineListener) list[i + 1]).vmException(e);
    }

    protected void fireVMException(Exception e) {
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == VirtualMachineListener.class)
                ((VirtualMachineListener) list[i + 1]).vmException(e);
    }

    protected void fireVMHalted() {
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == VirtualMachineListener.class)
                ((VirtualMachineListener) list[i + 1]).vmHalted();
    }

    protected class HaltException extends Exception {}

    protected class CallStackToPCListener implements CallStackListener, StackFrameListener {
        public void framePushed(CallStackEvent e) {
            e.getFrame().addStackFrameListener(this);
        }

        public void framePopped(CallStackEvent e) {
            e.getFrame().removeStackFrameListener(this);
        }

        public void pcChanged(StackFrameEvent e) {
            PCEvent pcEvent = new PCEvent(VirtualMachine.this, (StackFrame) e.getSource(), e.getPC());
            Object[] list = listeners.getListenerList();

            for (int i = list.length - 2; i >= 0; i -= 2)
                if (list[i] == PCListener.class)
                    ((PCListener) list[i + 1]).pcChanged(pcEvent);
        }

        public void framePopped(StackFrameEvent e) {}
    }

    protected class StepIntoListener implements PCListener {
        public void pcChanged(PCEvent e) {
            ((VirtualMachine) e.getSource()).removePCListener(this);
            wait = true;
        }
    }

    protected class StepOverListener extends StepOutListener {
        public void pcChanged(StackFrameEvent e) {
            ((StackFrame) e.getSource()).removeStackFrameListener(this);
            wait = true;
        }
    }

    protected class StepOutListener implements StackFrameListener {
        public void pcChanged(StackFrameEvent e) {}

        public void framePopped(StackFrameEvent e) {
            ((StackFrame) e.getSource()).removeStackFrameListener(this);
            wait = true;
        }
    }
}