package carmel.interpreter;

import carmel.tree.*;
import carmel.type.*;
import carmel.value.*;
import java.util.*;
import javax.swing.event.EventListenerList;

public class StackFrame {

    protected VirtualMachine vm;
    protected TreeConstructorOrMethod method;
    protected ProgramCounter pc;
    protected LocalVariableArray localVariables;
    protected OperandStack operandStack = new OperandStack();

    protected InterpreterVisitor interpreter = new InterpreterVisitor();

    protected boolean isConstructor = false;

    protected EventListenerList listeners = new EventListenerList();

    protected StackFrame(VirtualMachine vm, TreeConstructorOrMethod method) throws IllegalArgumentException {
        if (method.isAbstract())
            throw new IllegalArgumentException("Attempt to invoke an abstract method");
        if (method.isNative())
            throw new IllegalArgumentException("Attempt to interpret a native method");

        this.vm = vm;
        this.method = method;

        pc = method.instructionBlock.firstInstruction;
    }

    public StackFrame(VirtualMachine vm, TreeMethod method, List parameters) throws IllegalArgumentException {
        this(vm, method);

        if (!method.isStatic())
            throw new IllegalArgumentException("Attempt to invoke an instance method without a class value");

        localVariables = new LocalVariableArray(method.localVariableArraySize, parameters);
    }

    protected StackFrame(VirtualMachine vm, TreeConstructorOrMethod method, List parameters, ClassValue classValue) throws TypeException, IllegalArgumentException {
        this(vm, method);

        method.parentClass.checkAssignableFrom(classValue.getType());

        localVariables = new LocalVariableArray(method.localVariableArraySize, classValue, parameters);
    }

    public StackFrame(VirtualMachine vm, TreeConstructor method, List parameters, ClassValue classValue) throws TypeException, IllegalArgumentException {
        this(vm, (TreeConstructorOrMethod) method, parameters, classValue);

        isConstructor = true;
    }

    public StackFrame(VirtualMachine vm, TreeMethod method, List parameters, ClassValue classValue) throws TypeException, IllegalArgumentException {
        this(vm, (TreeConstructorOrMethod) method, parameters, classValue);

        if (method.isStatic())
            throw new IllegalArgumentException("Attempt to invoke a static method with a class value");
    }

    public TreeConstructorOrMethod getMethod() { return method; }

    public ProgramCounter getPC() { return pc; }

    public CarmelSource getCarmelSource() { return method.parentClass.parentPackage.source; }

    public OperandStack getOperandStack() { return operandStack; }

    public LocalVariableArray getLocalVariables() { return localVariables; }

    public void addStackFrameListener(StackFrameListener l) {
        listeners.add(StackFrameListener.class, l);
    }

    public void removeStackFrameListener(StackFrameListener l) {
        listeners.remove(StackFrameListener.class, l);
    }

    protected void firePCChanged() {
        StackFrameEvent e = new StackFrameEvent(this, pc);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == StackFrameListener.class)
                ((StackFrameListener) list[i + 1]).pcChanged(e);
    }

    protected void fireFramePopped() {
        StackFrameEvent e = new StackFrameEvent(this);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == StackFrameListener.class)
                ((StackFrameListener) list[i + 1]).framePopped(e);
    }

    public Value interpret() throws CarmelException, VerificationException, VMException, VirtualMachine.HaltException {
        vm.callStack.push(this);

        try {
            for (;;) {
                try {
                    firePCChanged();

                    // wait for vm to give us the go ahead
                    vm.waitForNextInstruction();

                    // set a restore point in case of a traceable exception
                    operandStack.restorePointStart();

                    // interpret the next instruction
                    ((Instruction) pc).visit(interpreter);

                    pc = pc.getNextPC();
                }
                catch (InterpreterVisitor.JumpException e) {
                    pc = e.instruction;
                }
                catch (CarmelException e) {
                    operandStack.restore();
                    TreeClass exceptionType = (TreeClass) e.getException().getType();

                    int index = ((Instruction) pc).blockIndex;

                    handlerSearch: {
                        // look for a handler
                        for (Iterator i = method.instructionBlock.handlers.iterator(); i.hasNext();) {
                            ExceptionHandler h = (ExceptionHandler) i.next();

                            if (h.fromIndex <= index && index <= h.toIndex && (h.catchType == null || h.catchType.isAssignableFrom(exceptionType))) {
                                // although strictly not an instruction, it is helpful to pause on the
                                // exception handler so that the user understands that an exception
                                // has been thrown.

                                // it is probably better for the user to see the state that led to the
                                // exception rather than the state afterwards.  the user will see this
                                // state at the next step anyway
                                pc = h;
                                firePCChanged();
                                vm.waitForNextInstruction();

                                operandStack.clear();
                                operandStack.push(e.getException());

                                pc = pc.getNextPC();

                                break handlerSearch;
                            }
                        }

                        // only let through runtime exceptions and those declared by the method
                        if (vm.classLoader.runtimeException.isAssignableFrom(exceptionType))
                            throw e;

                        for (Iterator i = method.getExceptionTypes().iterator(); i.hasNext();)
                            if (((TreeClass) i.next()).isAssignableFrom(exceptionType))
                                throw e;

                        throw new VerificationException(exceptionType.getName() + " must be caught or declared to be thrown");
                    }
                }
            }
        }
        catch (CarmelException e) {
            e.addFrame(this);
            vm.callStack.pop();
            fireFramePopped();
            throw e;
        }
        catch (VerificationException e) {
            operandStack.restore();
            throw e;
        }
        catch (InterpreterVisitor.ReturnException e) {
            vm.callStack.pop();
            fireFramePopped();
            return e.getValue();
        }
        catch (VirtualMachine.HaltException e) {
            throw e;
        }
        catch (VMException e) {
            throw e;
        }
        catch (Exception e) {
            throw new VMException("Exception encountered during intepreting", e);
        }
    }

    protected class InterpreterVisitor implements InstructionVisitor {

        public void visit(NopInstruction i) throws Exception {
            // do nothing
        }

        public void visit(PushInstruction i) throws Exception {
            operandStack.push(i.constant);
        }

        public void visit(PopInstruction i) throws Exception {
            operandStack.popWords(i.count);
        }

        public void visit(DupInstruction i) throws Exception {
            operandStack.dupWords(i.count, i.depth);
        }

        public void visit(SwapInstruction i) throws Exception {
            operandStack.swapWords(i.top, i.following);
        }

        public void visit(NumOpInstruction i) throws Exception {
            if (i.isUnary()) {
                operandStack.push(i.operandType.applyUnaryNumOp(i.operator, operandStack.popNumericValue(i.operandType), i.resultType));
            }
            else {
                NumericValue v2 = operandStack.popNumericValue(i.operandType);
                NumericValue v1 = operandStack.popNumericValue(i.operandType);

                try {
                    operandStack.push(i.operandType.applyBinaryNumOp(i.operator, v1, v2));
                }
                catch (ArithmeticException e) {
                    throw new CarmelException(vm.arithmeticException);
                }
            }
        }

        public void visit(LoadInstruction i) throws Exception {
            operandStack.push(localVariables.get(i.index, i.type));
        }

        public void visit(StoreInstruction i) throws Exception {
            localVariables.set(i.index, operandStack.pop(i.type));
        }

        public void visit(IncInstruction i) throws Exception {
            localVariables.set(i.index, i.type.applyBinaryNumOp(JCVMNumericType.ADD, (NumericValue) localVariables.get(i.index, i.type), i.value));
        }

        public void visit(GotoInstruction i) throws Exception {
            throw new JumpException(i.instruction);
        }

        public void visit(IfInstruction i) throws Exception {
            Value v2 = i.value == null ? operandStack.popValue(i.type) : i.value;
            Value v1 = operandStack.popValue(i.type);

            if (i.type.applyCondOp(i.operator, v1, v2))
                throw new JumpException(i.instruction);
        }

        public void visit(LookupSwitchInstruction i) throws Exception {
            Instruction instruction = (Instruction) i.switches.get(operandStack.popNumericValue(i.type));

            throw new JumpException(instruction == null ? i.instruction : instruction);
        }

        public void visit(TableSwitchInstruction i) throws Exception {
            try {
                throw new JumpException((Instruction) i.addresses.get(
                    ((IntValue) i.type.applyUnaryNumOp(JCVMNumericType.TO, operandStack.popNumericValue(i.type), JCVMIntType.TYPE)).getValue() - i.value
                ));
            }
            catch (IndexOutOfBoundsException e) {
                throw new JumpException(i.instruction);
            }
        }

        public void visit(NewArrayInstruction i) throws Exception {
            try {
                operandStack.push(new ArrayValue(vm.heap, i.type, operandStack.popShort().getValue()));
            }
            catch (NegativeArraySizeException e) {
                throw new CarmelException(vm.negativeArraySizeException);
            }
        }

        public void visit(NewClassInstruction i) throws Exception {
            operandStack.push(new ClassValue(vm.heap, i.classType));
        }

        public void visit(CheckCastInstruction i) throws Exception {
            if (!i.type.isAssignableFrom(((ReferenceValue) operandStack.peek(JCVMReferenceType.TYPE)).getType()))
                throw new CarmelException(vm.classCastException);

    //todo        if (v != NullValue.NULL)
    //            checkCast(i.type, v);
        }

        public void visit(InstanceOfInstruction i) throws Exception {
            operandStack.push(new ShortValue((short) (i.type.isAssignableFrom(operandStack.popReference().getType()) ? 1 : 0)));
        }

        public void visit(GetStaticInstruction i) throws Exception {
            operandStack.push(i.field.getValue());
        }

        public void visit(PutStaticInstruction i) throws Exception {
    //todo        checkPutStatic();
            i.field.setValue(operandStack.popValue(i.field.getType().getJCVMType()));
        }

        public void visit(GetFieldInstruction i) throws Exception {
            ClassValue c = i.thisField ? (ClassValue) localVariables.get(0) : popClassValue();

    //todo        checkGetField();

            operandStack.push(i.field.getValue(c));
        }

        public void visit(PutFieldInstruction i) throws Exception {
            ClassValue c = i.thisField ? (ClassValue) localVariables.get(0) : popClassValue();

    //todo        checkPutField();

            i.field.setValue(c, operandStack.popValue(i.field.getType().getJCVMType()));
        }

        protected ArrayValue popArrayValue() throws VerificationException, TypeException, CarmelException {
            ReferenceValue o = operandStack.popReference();

            checkNull(o);

            if (!(o instanceof ArrayValue))
                throw new TypeException("Array value expected, value of type " + o.getType().getName() + " found");

            return (ArrayValue) o;
        }

        protected ClassValue popClassValue() throws VerificationException, TypeException, CarmelException {
            ReferenceValue o = operandStack.popReference();

            checkNull(o);

            // make sure it is a class
            vm.classLoader.object.checkAssignableFrom(o.getType());

            return (ClassValue) o;
        }

        protected List popMethodParameters(List parameterTypes) throws VerificationException, TypeException {
            Value[] parameters = new Value[parameterTypes.size()];

            for (int index = parameters.length; --index >= 0;) {
                Type parameterType = (Type) parameterTypes.get(index);
                Value value = operandStack.popValue(parameterType.getJCVMType());

                if (!parameterType.isAssignableFrom(value.getType()))
                    throw new TypeException("Parameter " + index + " is of type " + parameterType.getName() + ", found incompatible type " + value.getType().getName());

                parameters[index] = value;
            }

            return new ArrayList(Arrays.asList(parameters));
        }

        protected void invokeMethod(TreeMethod method, List parameters, ClassValue classValue) throws Exception {
            Value v;

            if (method.isStatic()) {
                if (method.isNative())
                    throw new VMException("Native methods not yet implemented", new Exception());
                else
                    v = new StackFrame(vm, method, parameters).interpret();
            }
            else {
                // invocation of methods on uninitialised class instances is only
                // allowed from the constructor of the same class instance,
                // or one of one of the superclass constructors
                if (!classValue.isInit && isConstructor && !(classValue == localVariables.get(0) && method.getParentClass().isAssignableFrom(classValue.getType())))
                    throw new VerificationException("Attempt to invoke a method on an object that has not been initialised");

                if (method.isNative())
                    throw new VMException("Native methods not yet implemented", new Exception());
                else
                    v = new StackFrame(vm, method, parameters, classValue).interpret();
            }

            ResultType type = v == null ? VoidType.TYPE : (ResultType) v.getType();

            if (!method.getResultType().isAssignableFrom(type))
                throw new VerificationException("Method declared to return type " + method.getResultType().getName() + " but has returned a value of type " + type.getName());

            if (v != null) operandStack.push(v);
        }

        public void visit(InvokeConstructorInstruction i) throws Exception {
            List parameters = popMethodParameters(i.constructor.getParameterTypes());

            ClassValue classValue = popClassValue();

            if (classValue.isInit) throw new VerificationException("Attempt to call constructor for a second time");

            if (new StackFrame(vm, i.constructor, parameters, classValue).interpret() != null)
                throw new VerificationException("Constructor cannot return a value");

            // only set isInit if we are the original constructor call,
            // not a nested constructor call
            if (!(isConstructor && classValue == localVariables.get(0)))
                classValue.isInit = true;
        }

        public void visit(InvokeDefiniteMethodInstruction i) throws Exception {
            List parameters = popMethodParameters(i.method.getParameterTypes());

            if (i.method.isStatic()) {
                // todo: checks
    //todo            checkInvokeDefinite();

                invokeMethod(i.method, parameters, null);
            }
            else {
                invokeMethod(i.method, parameters, popClassValue());
            }
        }

        public void visit(InvokeVirtualInstruction i) throws Exception {
            List parameters = popMethodParameters(i.methodID.types);

            ClassValue v = popClassValue();
            TreeClass type = (TreeClass) v.getType();

            // todo: checks
    //todo        checkInvokeVirtual();

            try {
                invokeMethod(type.getMethod(i.methodID), parameters, v);
            }
            catch (NoSuchMethodException e) {
                throw new VerificationException(e.getMessage());
            }
        }

        public void visit(InvokeInterfaceInstruction i) throws Exception {
            List parameters = popMethodParameters(i.methodID.types);

            ClassValue v = popClassValue();
            TreeClass type = (TreeClass) v.getType();

            // todo: checks
            i.parentInterface.checkAssignableFrom(type);

            try {
                invokeMethod(type.getMethod(i.methodID), parameters, v);
            }
            catch (NoSuchMethodException e) {
                throw new VerificationException(e.getMessage());
            }
        }

        public void visit(ReturnInstruction i) throws Exception {
            if (i.type == null) throw new ReturnException();

            Value v = operandStack.popValue(i.type);

            // constructors return void, so we know that it's a method
            ((TreeMethod) method).getResultType().checkAssignableFrom(v.getType());

            throw new ReturnException(v);
        }

        public void visit(ArrayLengthInstruction i) throws Exception {
            operandStack.push(new ShortValue(popArrayValue().getLength()));
        }

        public void visit(ArrayLoadInstruction i) throws Exception {
    //todo        checkArrayLoad();
            short index = operandStack.popShort().getValue();

            try {
                operandStack.push(popArrayValue().getValueAt(index));
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new CarmelException(vm.arrayIndexOutOfBoundsException);
            }
        }

        public void visit(ArrayStoreInstruction i) throws Exception {
            StackEntry v = operandStack.pop();
            short index = operandStack.popShort().getValue();

            ArrayValue a = popArrayValue();

            a.getComponentType().getJCVMType().checkType(v);
    //todo        checkArrayStore();

            try {
                a.setValueAt(index, (Value) v);
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new CarmelException(vm.arrayIndexOutOfBoundsException);
            }
        }

        public void visit(ThrowInstruction i) throws Exception {
            ClassValue c = popClassValue();

            vm.classLoader.throwable.checkAssignableFrom(c.getType());

            //todo
    //todo        checkThrow();

            throw new CarmelException(c);
        }

        public void visit(JsrInstruction i) throws Exception {
            //todo: i.next must exist!
            operandStack.push(i.next);

            throw new JumpException(i.instruction);
        }

        public void visit(RetInstruction i) throws Exception {
            throw new JumpException((Instruction) localVariables.get(i.index, JCVMReturnAddressType.TYPE));
        }

        protected void checkNull(ReferenceValue v) throws CarmelException {
            if (v.getType() == NullType.TYPE)
                throw new CarmelException(vm.nullPointerException);
        }

        protected class JumpException extends Exception {
            Instruction instruction;

            JumpException(Instruction instruction) {
                this.instruction = instruction;
            }
        }

        protected class ReturnException extends Exception {
            Value value = null;

            ReturnException() {}
            ReturnException(Value value) { this.value = value; }

            Value getValue() { return value; }
        }
    }
}