package carmel.interpreter;

import carmel.type.*;
import carmel.value.*;
import java.util.*;
import javax.swing.event.*;

public class OperandStack implements StackEntryList {

    protected LinkedList list = new LinkedList();

    protected LinkedList restoreList = new LinkedList();
    protected EventListenerList listeners = new EventListenerList();

    protected boolean restore = false;

    public void push(StackEntry o) throws NullPointerException {
        restore = false;

        if (o == null) throw new NullPointerException("Null is not a valid stack value");

        JCVMType type = o.getJCVMType();

        if (type == JCVMByteType.TYPE) {
            list.addFirst(((ByteValue) o).toShort());
            fireEntriesPushed(1);
        }
        else {
            if (type.isDoubleWord()) {
                list.addFirst(BottomValue.BOTTOM);
                list.addFirst(o);
                fireEntriesPushed(2);
            }
            else {
                list.addFirst(o);
                fireEntriesPushed(1);
            }
        }
    }

    public StackEntry pop() throws VerificationException {
        try {
            StackEntry o = (StackEntry) list.removeFirst();
            if (restore) restoreList.add(o);

            if (!list.isEmpty() && list.getFirst() == BottomValue.BOTTOM) {
                if (restore) restoreList.add(BottomValue.BOTTOM);
                list.removeFirst();
                fireEntriesPopped(2);
            }
            else
                fireEntriesPopped(1);

            return o;
        }
        catch (NoSuchElementException e) {
            throw new VerificationException("Attempt to pop a value from an empty operand stack");
        }
    }

    public StackEntry pop(JCVMType type) throws TypeException, VerificationException {
        if (type == JCVMByteType.TYPE)
            return ByteValue.fromShort(popShort());

        StackEntry o = pop();
        type.checkType(o);
        return o;
    }

    public Value popValue(JCVMOperandType type) throws TypeException, VerificationException {
        return (Value) pop(type);
    }

    public NumericValue popNumericValue(JCVMNumericType type) throws TypeException, VerificationException {
        return (NumericValue) pop(type);
    }

    public ReferenceValue popReference() throws TypeException, VerificationException {
        return (ReferenceValue) pop(JCVMReferenceType.TYPE);
    }

    public ShortValue popShort() throws TypeException, VerificationException {
        return (ShortValue) pop(JCVMShortType.TYPE);
    }

    public StackEntry peek() throws VerificationException {
        try {
            return (StackEntry) list.getFirst();
        }
        catch (NoSuchElementException e) {
            throw new VerificationException("Attempt to peek at a value on an empty operand stack");
        }
    }

    public StackEntry peek(JCVMType type) throws TypeException, VerificationException {
        if (JCVMByteType.TYPE.equals(type)) {
            return ByteValue.fromShort((ShortValue) peek(JCVMShortType.TYPE));
        }
        else {
            StackEntry o = peek();
            type.checkType(o);
            return o;
        }
    }

    public boolean isEmpty() {
        return list.isEmpty();
    }

    public void clear() {
        list.clear();
    }

    public int getSizeInWords() {
        return list.size();
    }

    public void popWords(int numWords) throws VerificationException {
        if (numWords == 0) return;

        checkLastIndex(numWords);

        restore = false;

        list.subList(0, numWords).clear();

        fireEntriesPopped(numWords);
    }

    public void dupWords(int numWords, int depth) throws VerificationException {
        if (numWords == 0) return;

        checkLastIndex(numWords);

        restore = false;

        List subList = list.subList(0, numWords);

        // 0 < depth < numWords prohibited by parser

        list.addAll(depth, new LinkedList(subList));

        fireEntriesPushed(numWords);

        // todo: the event isn't quite suited to this operation, perhaps
        // a range would be better, ie (depth, numWords)
        if (depth != 0)
            fireEntriesChanged(depth + numWords);
    }

    public void swapWords(int n1, int n2) throws VerificationException {
        if (n1 + n2 == 0) return;

        checkLastIndex(n1 + n2);

        restore = false;

        // get S1
        List subList = list.subList(0, n1);
        List subListCopy = new LinkedList(subList);

        // remove S1
        subList.clear();

        // add after S2
        list.addAll(n2, subListCopy);

        fireEntriesChanged(n1 + n2);
    }

    public void restorePointStart() {
        restore = true;
        restoreList.clear();
    }

    public void restore() {
        if (!restore) throw new IllegalStateException("Cannot restore after push, popWords, dupWords or swapWords");

        for (Iterator i = restoreList.iterator(); i.hasNext();)
            push((StackEntry) i.next());

        restore = false;
    }

    public void addStackEntryListListener(StackEntryListListener l) {
        listeners.add(OperandStackListener.class, l);
    }

    public void removeStackEntryListListener(StackEntryListListener l) {
        listeners.remove(OperandStackListener.class, l);
    }

    protected void fireEntriesPopped(int numWords) {
        OperandStackEvent e = new OperandStackEvent(this, numWords);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == OperandStackListener.class)
                ((OperandStackListener) list[i + 1]).entriesPopped(e);
    }

    protected void fireEntriesPushed(int numWords) {
        OperandStackEvent e = new OperandStackEvent(this, numWords);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == OperandStackListener.class)
                ((OperandStackListener) list[i + 1]).entriesPushed(e);
    }

    protected void fireEntriesChanged(int numWords) {
        OperandStackEvent e = new OperandStackEvent(this, numWords);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == OperandStackListener.class)
                ((OperandStackListener) list[i + 1]).entriesChanged(e);
    }

    protected void checkLastIndex(int index) throws VerificationException {
        if (index > list.size())
            throw new VerificationException("Stack size too small");
        if (index != list.size() && list.get(index) == BottomValue.BOTTOM)
            throw new VerificationException("Attempt to break apart a double word");
    }

    public int getEntryListSize() { return list.size(); }

    public StackEntry getEntryNoVerification(int index) {
        return (StackEntry) list.get(index);
    }
}