package carmel.interpreter;

import carmel.type.*;
import carmel.value.*;
import javax.swing.event.EventListenerList;
import java.util.*;

public class LocalVariableArray implements StackEntryList {

    protected StackEntry[] variables;

    protected EventListenerList listeners = new EventListenerList();

    protected boolean firstEntryProtected = false;

    public LocalVariableArray(int size) {
        variables = new StackEntry[size];
    }

    public LocalVariableArray(int size, List parameterEntries)  {
        this(size);
        fireVariablesChanged(0, addList(0, parameterEntries));
    }

    public LocalVariableArray(int size, ClassValue classValue, List parameterEntries) {
        this(size);
        firstEntryProtected = true;
        variables[0] = classValue;
        fireVariablesChanged(0, addList(1, parameterEntries));
    }

    protected int addList(int index, List list) {
        try {
            for (Iterator i = list.iterator(); i.hasNext();) {
                StackEntry o = (StackEntry) i.next();

                set(index++, o);

                if (o.getJCVMType().isDoubleWord()) index++;
            }

            return index;
        }
        catch (VerificationException e) {
            throw new Error("Unexpected VerificationException in addList");
        }
    }

    public StackEntry get(int index) throws VerificationException {
//        try {
            StackEntry o = variables[index];

            if (o == null) throw new VerificationException("Attempt to access an uninitialised local variable");

            if (BottomValue.BOTTOM.equals(o)) throw new VerificationException("Attempt to access second word of a double word local variable");

            return o;
//        }
//        catch (ArrayIndexOutOfBoundsException e) {
//            throw new VerificationException("Local variable index out of bounds: Index: " + index + " Size: " + variables.length);
//        }
    }

    public StackEntry get(int index, JCVMType type) throws VerificationException, TypeException {
        if (type == JCVMByteType.TYPE)
            return ByteValue.fromShort((ShortValue) get(index, JCVMShortType.TYPE));

        StackEntry o = get(index);
        type.checkType(o);
        return o;
    }

    public void set(int index, StackEntry o) throws VerificationException {
        if (o == null) throw new NullPointerException("Null is not a valid local variable value");

        if (firstEntryProtected && index == 0) throw new VerificationException("Class value at index 0 cannot be modified");

        // invalidate the int we have half overwritten
        if (variables[index] == BottomValue.BOTTOM) variables[index - 1] = BottomValue.BOTTOM;

        JCVMType type = o.getJCVMType();
        if (type == JCVMByteType.TYPE) o = ((ByteValue) o).toShort();

//        try {
            variables[index] = o;

            if (type.isDoubleWord()) {
                variables[index + 1] = BottomValue.BOTTOM;
                fireVariablesChanged(index, index + 1);
            }
            else
                fireVariablesChanged(index, index);
//        }
//        catch (ArrayIndexOutOfBoundsException e) {
//            throw new VerificationException("Local variable index out of bounds: Index: " + index + " Size: " + variables.length);
//        }
    }

    public int getSize() { return variables.length; }

    public void addStackEntryListListener(StackEntryListListener l) {
        listeners.add(LocalVariableArrayListener.class, l);
    }

    public void removeStackEntryListListener(StackEntryListListener l) {
        listeners.remove(LocalVariableArrayListener.class, l);
    }

    protected void fireVariablesChanged(int firstIndex, int lastIndex) {
        LocalVariableArrayEvent e = new LocalVariableArrayEvent(this, firstIndex, lastIndex);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == LocalVariableArrayListener.class)
                ((LocalVariableArrayListener) list[i + 1]).variablesChanged(e);
    }

    public int getEntryListSize() {
        return getSize();
    }

    public StackEntry getEntryNoVerification(int index) {
        return variables[index];
    }
}