package carmel.interpreter;

import carmel.value.NonNullReferenceValue;
import java.util.*;
import javax.swing.event.EventListenerList;

public class Heap {

    protected short nextFreeLocation = 0;
    // as lookups almost never occur, it is more important to be
    // able to easily iterate over the map in key order
    protected ArrayList mappings = new ArrayList();

    protected EventListenerList listeners = new EventListenerList();

    public short allocate(NonNullReferenceValue value) {
        short location = nextFreeLocation;
        nextFreeLocation += value.getSizeInBytes();
        mappings.add(new Mapping(location, value));
        fireHeapAllocationEvent(mappings.size() - 1);
        return location;
    }

    public NonNullReferenceValue lookup(short location) {
        int index = getIndexForLocation(location);

        return (index == -1) ? null : ((Mapping) mappings.get(index)).value;
    }

    public int getIndexForLocation(short location) {
        int index = Collections.binarySearch(mappings, new Mapping(location));

        return (index < 0) ? -1 : index;
    }

    public short getSizeInBytes() { return nextFreeLocation; }

    public int getMappingCount() { return mappings.size(); }

    public Mapping getMapping(int index) {
        return (Mapping) mappings.get(index);
    }

    public Collection getMappings() { return mappings; }

    public void addHeapListener(HeapListener l) {
        listeners.add(HeapListener.class, l);
    }

    public void removeHeapListener(HeapListener l) {
        listeners.remove(HeapListener.class, l);
    }

    protected void fireHeapAllocationEvent(int index) {
        HeapEvent e = new HeapEvent(this, index);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == HeapListener.class)
                ((HeapListener) list[i + 1]).newAllocation(e);
    }

    public class Mapping implements Comparable {
        protected short location;
        protected NonNullReferenceValue value = null;

        protected Mapping(short location) {
            this.location = location;
        }

        protected Mapping(short location, NonNullReferenceValue value) {
            this.location = location;
            this.value = value;
        }

        public short getLocation() { return location; }
        public NonNullReferenceValue getValue() { return value; }

        public int compareTo(Object o) {
            return location - ((Mapping) o).location;
        }

        public boolean equals(Object o) {
            return location == ((Mapping) o).location;
        }

        public int hashCode() {
            return location;
        }
    }
}