package carmel.interpreter;

import carmel.parser.*;
import carmel.tree.*;
import carmel.type.*;
import carmel.value.ClassValue;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.event.EventListenerList;

public class CarmelClassLoader {

    // number of link passes after a package is parsed
    // Note:
    // if you add a link pass, increment this value and add the link pass
    // to the switch statement in linkPackage(String, int)
    // everything else should (hopefully) work
    protected static final int LINK_PASSES = 2;

    public final TreeClass object, throwable, runtimeException;

    // first element for packages which have only been parsed
    // second element for packages linked by pass 1
    // third element for packages linked by pass 2
    protected HashMap[] pendingPackages;

    protected HashMap packages = new HashMap();
    protected HashMap packagesByAID = new HashMap();

    protected Heap heap;

    protected EventListenerList listeners = new EventListenerList();

    public static String getPackageFilename(String packageName) {
        // this works even if there is no '.' in packageName
        return (packageName == null) ? "carmel/null.cml" : packageName.replace('.', '/') + "/carmel/" + packageName.substring(packageName.lastIndexOf('.') + 1) + ".cml";
    }

    public CarmelClassLoader(Heap heap) throws ClassNotFoundException, LexOrParseException {
        this.heap = heap;

        pendingPackages = new HashMap[LINK_PASSES + 1];

        for (int i = 0; i <= LINK_PASSES; i++)
            pendingPackages[i] = new HashMap();

        // these classes are needed in order to parse & execute the carmel language
        object =            (TreeClass) loadClass("java.lang.Object");
        throwable =         (TreeClass) loadClass("java.lang.Throwable");
        runtimeException =  (TreeClass) loadClass("java.lang.RuntimeException");
    }

    public Collection getPackages() {
        return packages.values();
    }

    public TreeClassOrInterface loadClass(String name) throws ClassNotFoundException, LexOrParseException {
        TreeClassOrInterface c = loadClass(name, LINK_PASSES + 1);

        commit();

        return c;
    }

    public TreeClassOrInterface loadClass(String name, int linkPass) throws ClassNotFoundException, LexOrParseException {
        int index = name.lastIndexOf('.');

        // unnamed package represented as null
        String packageName = (index == -1) ? null : name.substring(0, index);
        String className = name.substring(index + 1);

        try {
            TreeClassOrInterface c = loadPackage(packageName, linkPass).getClassOrInterface(className);

            if (c != null) return c;
        }
        catch (PackageNotFoundException e) {}

        abandon();

        throw new ClassNotFoundException("Class " + name + " not found");
    }

    public TreePackage loadPackage(String name) throws LexOrParseException, PackageNotFoundException {
        TreePackage p = loadPackage(name, LINK_PASSES + 1);
        commit();
        return p;
    }

    public TreePackage loadPackage(String name, int linkPass) throws LexOrParseException, PackageNotFoundException {
        TreePackage p;

        // if package is not loaded, load to level 0
        packageSearch: {
            p = (TreePackage) packages.get(name);
            if (p != null) return p;

            for (int i = LINK_PASSES; i >= linkPass; i--) {
                p = (TreePackage) pendingPackages[i].get(name);
                if (p != null) return p;
            }

            for (int i = linkPass - 1; i >= 0; i--) {
                p = (TreePackage) pendingPackages[i].get(name);
                if (p != null) break packageSearch;
            }

            if (linkPass > 0) p = linkPackage(name, 0);
        }

        // first link everything (including p) from 0..linkPass-2 to linkPass-1
        for (int i = 0; i <= linkPass - 2; i++) {
            while (true) {
                Collection c = pendingPackages[i].values();

                if (c.isEmpty()) break;

                linkPackage(((TreePackage) c.iterator().next()).getName(), i + 1);
            }
        }

        if (linkPass == LINK_PASSES + 1)
            return p;
        else
            return linkPackage(name, linkPass);
    }

    protected void abandon() {
        for (int i = 0; i <= LINK_PASSES; i++) {
            pendingPackages[i].clear();
        }
    }

    protected TreePackage linkPackage(String name, int linkPass) throws PackageNotFoundException, LexOrParseException {
        try {
            TreePackage p = null;

            if (linkPass > 0) p = (TreePackage) pendingPackages[linkPass - 1].get(name);

            switch (linkPass) {
                case 0:
                    // todo: this doesn't work with Web Start: it may have to be implemented
                    // explicitly
                    URL url = getClass().getClassLoader().getResource(getPackageFilename(name));

                    if (url == null) throw new PackageNotFoundException();

                    try {
                        p = new Parser(heap, URLCarmelSource.createCarmelSource(url)).parsePackage();
                        // todo: check with all loaded and half-loaded packages for name/aid conflicts
                    }
                    catch (FileNotFoundException e) {
                        e.printStackTrace();
                        throw new PackageNotFoundException();
                    }
                    catch (IOException e) {
                        e.printStackTrace();
                        throw new PackageNotFoundException();
                    }

                    break;
                case 1:
                    new LinkPass1Visitor(this).visitPackage(p);
                    break;
                case 2:
                    new LinkPass2Visitor(this).visitPackage(p);
                    break;
                //
                // add new link passes here
                //
                default:
                    throw new Error("Invalid link pass: " + linkPass);
            }

            if (linkPass > 0) pendingPackages[linkPass - 1].remove(name);
            pendingPackages[linkPass].put(name, p);

            return p;
        }
        catch (PackageNotFoundException e) {
            abandon();
            throw e;
        }
        catch (LexOrParseException e) {
            abandon();
            throw e;
        }
    }

    protected void commit() {
        packages.putAll(pendingPackages[LINK_PASSES]);

        for (Iterator i = pendingPackages[LINK_PASSES].values().iterator(); i.hasNext();)
            firePackageLoaded((TreePackage) i.next());

        abandon();
    }

    public List loadMultiPackageFile(CarmelSource source) throws IOException, LexOrParseException {
        List trees = new Parser(heap, source).parsePackages();

        for (Iterator i = trees.iterator(); i.hasNext();) {
            TreePackage p = (TreePackage) i.next();

            if (packages.get(p.getName()) != null) {
                ParseException e = new ParseException("A package with name " + p.getName() + " has already been declared", p.token);
                e.setSource(source);
                throw e;
            }

//            TreePackage p2 = (TreePackage) packagesByAID.get(p.getAID());
//
//            if (p2 != null)
//                throw new ParseException("Package " + p.getName() + " shares AID with loaded package " + p2.getName(), p.token);
//
            pendingPackages[0].put(p.getName(), p);
        }

        try {
            for (Iterator i = trees.iterator(); i.hasNext();)
                loadPackage(((TreePackage) i.next()).getName(), LINK_PASSES + 1);
        }
        catch (PackageNotFoundException e) {
            abandon();
            throw new InternalError("unexpected PackageNotFoundException");
        }

        commit();

        return trees;
    }

    public void addClassLoaderListener(ClassLoaderListener l) {
        listeners.add(ClassLoaderListener.class, l);
    }

    public void removeClassLoaderListener(ClassLoaderListener l) {
        listeners.remove(ClassLoaderListener.class, l);
    }

    protected void firePackageLoaded(TreePackage p) {
        ClassLoaderEvent e = new ClassLoaderEvent(this, p);
        Object[] list = listeners.getListenerList();

        for (int i = list.length - 2; i >= 0; i -= 2)
            if (list[i] == ClassLoaderListener.class)
                ((ClassLoaderListener) list[i + 1]).packageLoaded(e);
    }
}