package carmel.gui;

import carmel.interpreter.*;
import carmel.parser.LexOrParseException;
import carmel.tree.*;
import carmel.value.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.NoSuchElementException;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.text.BadLocationException;
import javax.swing.tree.*;

public class VMFrame extends JFrame {

    protected Action loadPackageAction, loadFileAction, setClassPathAction, runMethodAction;
    protected Action stepIntoAction, stepOverAction, stepOutAction, haltAction, closeAction;

    protected CallStackTableModel callStackTableModel = new CallStackTableModel();
    protected SingleListSelectionModel callStackTableSelectionModel = new SingleListSelectionModel();
    protected OperandStackTableModel operandStackTableModel = new OperandStackTableModel();
    protected LocalVariableArrayTableModel localVariablesTableModel = new LocalVariableArrayTableModel();

    protected HeapTableModel heapTableModel = new HeapTableModel();
    protected SingleListSelectionModel heapTableSelectionModel = new SingleListSelectionModel();
    protected ClassValueTableModel classValueTableModel = new ClassValueTableModel();
    protected ArrayValueTableModel arrayValueTableModel = new ArrayValueTableModel();

    protected ClassLoaderTreeModel classTreeModel = new ClassLoaderTreeModel();
    protected StaticFieldTableModel staticFieldsTableModel = new StaticFieldTableModel();

    protected ClassPathEditorPanel classPathEditorPanel = new ClassPathEditorPanel();
    protected LoadCarmelSourcePanel loadCarmelSourcePanel = new LoadCarmelSourcePanel();

    // JBuilder generated swing components
    TitledBorder heapBorder;
    TitledBorder staticFieldsBorder;
    TitledBorder callStackBorder;
    TitledBorder localVariablesBorder;
    TitledBorder operandStackBorder;
    TitledBorder fileViewerBorder;
    BorderLayout borderLayout3 = new BorderLayout();
    JSplitPane splitPane = new JSplitPane();
    JScrollPane callStackScrollPane = new JScrollPane();
    JSplitPane callStackSplitPane = new JSplitPane();
    JScrollPane operandStackScrollPane = new JScrollPane();
    JTable callStackTable = new JTable();
    JTable localVariablesTable = new JTable();
    JPanel framePanel = new JPanel();
    JTable operandStackTable = new JTable();
    JPanel vmStatePanel = new JPanel();
    GridBagLayout gridBagLayout3 = new GridBagLayout();
    GridBagLayout gridBagLayout2 = new GridBagLayout();
    JScrollPane localVariablesScrollPane = new JScrollPane();
    BorderLayout borderLayout1 = new BorderLayout();
    BorderLayout borderLayout2 = new BorderLayout();
    JLabel heapInfoLabel = new JLabel();
    JScrollPane heapTableScrollPane = new JScrollPane();
    JPanel heapPanel = new JPanel();
    JSplitPane heapSplitPane = new JSplitPane();
    JTable heapTable = new JTable();
    JSplitPane sourceSplitPane = new JSplitPane();
    CarmelSourcePanel carmelSourcePanel = new CarmelSourcePanel();
    JScrollPane staticFieldsScrollPane = new JScrollPane();
    JScrollPane classTreeScrollPane = new JScrollPane();
    JSplitPane staticFieldsSplitPane = new JSplitPane();
    JTable staticFieldsTable = new JTable();
    JTree classTree = new JTree();
    JPanel buttonPanel = new JPanel();
    JButton stepIntoButton = new JButton();
    JButton closeButton = new JButton();
    JButton stepOutButton = new JButton();
    JButton stepOverButton = new JButton();
    JPanel runButtonPanel = new JPanel();
    CardLayout buttonPanelCardLayout = new CardLayout();
    JPanel loadButtonPanel = new JPanel();
    JButton loadPackageButton = new JButton();
    JButton loadFileButton = new JButton();
    JButton setClassPathButton = new JButton();
    JButton runMethodButton = new JButton();
    JButton close2Button = new JButton();
    JButton haltButton = new JButton();
    JTable valueExplorerTable = new JTable();
    JScrollPane valueExplorerScrollPane = new JScrollPane();

    public VMFrame(String fileName) {
        this();
        loadCarmelSourcePanel.setFileName(fileName);
    }

    public VMFrame() {
        super("Carmel Interpreter");

        try {
            final VirtualMachine vm = new VirtualMachine();

            try {
                jbInit();
            }
            catch(Exception e) {
                e.printStackTrace();
            }

            addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    closeWindow(vm);
                }
            });

            loadPackageAction = new AbstractAction("Load Package...") {
                public void actionPerformed(ActionEvent event) {
                    String name = JOptionPane.showInputDialog(null, "Enter package name", "Load package", JOptionPane.QUESTION_MESSAGE);

                    if (name != null) {
                        try {
                            TreePackage p = vm.getClassLoader().loadPackage(name);
                            classTree.getSelectionModel().setSelectionPath(new TreePath(classTreeModel.getPackageNode(p).getPath()));
                            classTree.expandPath(null);
                        }
                        catch (LexOrParseException e) {
                            new SourceErrorDialog(null, e, "Compile error").show();
                        }
                        catch (PackageNotFoundException e) {
                            JOptionPane.showMessageDialog(null, "Package not found.  Ensure that its .cml file is in the class path.", "Package not found", JOptionPane.ERROR_MESSAGE);
                        }
                    }
                }
            };
            loadPackageButton.setAction(loadPackageAction);

            loadFileAction = new AbstractAction("Load File...") {
                public void actionPerformed(ActionEvent event) {
                    if (JOptionPane.showConfirmDialog(null, loadCarmelSourcePanel, "Open Carmel Source", JOptionPane.OK_CANCEL_OPTION) == JOptionPane.OK_OPTION) {
                        try {
                            TreePackage p = (TreePackage) vm.getClassLoader().loadMultiPackageFile(loadCarmelSourcePanel.getCarmelSource()).get(0);
                            classTree.getSelectionModel().setSelectionPath(new TreePath(classTreeModel.getPackageNode(p).getPath()));
                        }
                        catch (IOException e) {
                            JOptionPane.showMessageDialog(null, "Error reading file: " + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
                        }
                        catch (LexOrParseException e) {
                            new SourceErrorDialog(null, e, "Compile error").show();
                        }
                    }
                }
            };
            loadFileButton.setAction(loadFileAction);

            setClassPathAction = new AbstractAction("Set classpath...") {
                public void actionPerformed(ActionEvent e) {
                    classPathEditorPanel.refreshClassPath();

                    if (JOptionPane.showConfirmDialog(null, classPathEditorPanel, "Set classpath", JOptionPane.OK_CANCEL_OPTION) == JOptionPane.OK_OPTION) {
                        classPathEditorPanel.commitChange();
                    }
                }
            };
            setClassPathButton.setAction(setClassPathAction);

            runMethodAction = new AbstractAction("Run Method") {
                public void actionPerformed(ActionEvent event) {
                    TreePath selectedPath = classTree.getAnchorSelectionPath();

                    if (selectedPath != null) {
                        Object selectedNode = selectedPath.getLastPathComponent();

                        if (selectedNode instanceof ClassLoaderTreeModel.MethodNode) {
                            TreeMethod method = ((ClassLoaderTreeModel.MethodNode) selectedNode).getMethod();

                            try {
                                if (method.isStatic()) {
                                    vm.invokeStaticMethod(method);
                                    buttonPanelCardLayout.show(buttonPanel, "runButtonPanel");
                                }
                                else {
                                    int index = heapTableSelectionModel.getSelectedIndex();

                                    if (index != -1) {
                                        NonNullReferenceValue value = vm.getHeap().getMapping(index).getValue();

                                        if (value instanceof ClassValue) {
                                            vm.invokeInstanceMethod(method, (ClassValue) value);
                                            buttonPanelCardLayout.show(buttonPanel, "runButtonPanel");
                                        }

                                        return;
                                    }

                                    JOptionPane.showMessageDialog(null, "Please select a class instance from the heap on which to execute the method", "Error", JOptionPane.ERROR_MESSAGE);
                                }
                            }
                            catch (IllegalArgumentException e) {
                                buttonPanelCardLayout.show(buttonPanel, "loadButtonPanel");
                                JOptionPane.showMessageDialog(null, "Error invoking method: " + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
                            }

                            return;
                        }
                        else if (selectedNode instanceof ClassLoaderTreeModel.ConstructorNode) {
                            TreeConstructor c = ((ClassLoaderTreeModel.ConstructorNode) selectedNode).getConstructor();

                            try {
                                vm.invokeConstructor(c);
                                buttonPanelCardLayout.show(buttonPanel, "runButtonPanel");
                            }
                            catch (IllegalArgumentException e) {
                                JOptionPane.showMessageDialog(null, "Error invoking constructor: " + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
                            }

                            return;
                        }
                    }

                    JOptionPane.showMessageDialog(null, "Please select a method or constructor from the \"Loaded Classes\" tree", "Error", JOptionPane.ERROR_MESSAGE);
                }
            };
            runMethodButton.setAction(runMethodAction);

            stepIntoAction = new AbstractAction("Step Into") {
                public void actionPerformed(ActionEvent e) {
                    vm.stepInto();
                }
            };
            stepIntoButton.setAction(stepIntoAction);

            stepOverAction = new AbstractAction("Step Over") {
                public void actionPerformed(ActionEvent e) {
                    vm.stepOver();
                }
            };
            stepOverButton.setAction(stepOverAction);

            stepOutAction = new AbstractAction("Step Out") {
                public void actionPerformed(ActionEvent e) {
                    vm.stepOut();
                }
            };
            stepOutButton.setAction(stepOutAction);

            haltAction = new AbstractAction("Halt") {
                public void actionPerformed(ActionEvent e) {
                    vm.halt();
                }
            };
            haltButton.setAction(haltAction);

            closeAction = new AbstractAction("Close") {
                public void actionPerformed(ActionEvent e) {
                    closeWindow(vm);
                }
            };
            closeButton.setAction(closeAction);
            close2Button.setAction(closeAction);

            callStackTable.addMouseListener(new MouseAdapter() {
                public void mouseClicked(MouseEvent event) {
                    try {
                        updateFileTextArea(vm.getCallStack().getFrame(((JTable) event.getSource()).rowAtPoint(event.getPoint())));
                    }
                    catch (IndexOutOfBoundsException e) {}
                }
            });

            callStackTableSelectionModel.addListSelectionListener(new ListSelectionListener() {
                public void valueChanged(ListSelectionEvent e) {
                    final int index = ((ListSelectionModel) e.getSource()).getAnchorSelectionIndex();

                    // we are not in the event dispatch thread!
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            try {
                                StackFrame frame = vm.getCallStack().getFrame(index);

                                operandStackTableModel.setOperandStack(frame.getOperandStack());
                                localVariablesTableModel.setStackEntryList(frame.getLocalVariables());

                                updateFileTextArea(frame);
                            }
                            catch (IndexOutOfBoundsException ex) {
                                operandStackTableModel.setOperandStack(null);
                                localVariablesTableModel.setStackEntryList(null);
                            }
                        }
                    });
                }
            });

            classTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

            classTree.addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    ((ClassLoaderTreeModel.Node) e.getPath().getLastPathComponent()).visit(new ClassLoaderNodeVisitor() {
                        protected void highlight(int lineNumber) {
                            try {
                                carmelSourcePanel.highlightLine(lineNumber - 1);
                            }
                            catch (BadLocationException e) {}
                        }

                        public void visitPackage(TreePackage p, boolean highlight) {
                            carmelSourcePanel.setCarmelSource(p.getCarmelSource());
                            if (highlight) highlight(p.getLineNumber());
                        }

                        public void visit(ClassLoaderTreeModel.PackageNode n) {
                            visitPackage(n.getPackage(), true);
                            staticFieldsTableModel.setClassOrInterface(null);
                        }

                        public void visitClassOrInterface(TreeClassOrInterface c, boolean highlight) {
                            visitPackage(c.getPackage(), false);
                            if (highlight) highlight(c.getLineNumber());
                            staticFieldsTableModel.setClassOrInterface(c);
                        }

                        public void visit(ClassLoaderTreeModel.ClassNode n) {
                            visitClassOrInterface(n.getClassOrInterface(), true);
                        }

                        public void visit(ClassLoaderTreeModel.InterfaceNode n) {
                            visitClassOrInterface(n.getClassOrInterface(), true);
                        }

                        public void visitConstructorOrMethod(TreeConstructorOrMethod m) {
                            visitClassOrInterface(m.getParentClass(), false);
                            highlight(m.getLineNumber());
                        }

                        public void visit(ClassLoaderTreeModel.ConstructorNode n) {
                            visitConstructorOrMethod(n.getConstructorOrMethod());
                        }

                        public void visit(ClassLoaderTreeModel.MethodNode n) {
                            visitConstructorOrMethod(n.getConstructorOrMethod());
                        }
                    });
                }
            });

            heapTableSelectionModel.addListSelectionListener(new ListSelectionListener() {
                public void valueChanged(ListSelectionEvent e) {
                    final int index = heapTableSelectionModel.getSelectedIndex();

                    if (index == -1) {
                        valueExplorerTable.setModel(null);
                    }
                    else {
                        NonNullReferenceValue value = vm.getHeap().getMapping(index).getValue();

                        if (value instanceof ClassValue) {
                            classValueTableModel.setClassValue((ClassValue) value);
                            valueExplorerTable.setModel(classValueTableModel);
                        }
                        else if (value instanceof ArrayValue) {
                            arrayValueTableModel.setStackEntryList((ArrayValue) value);
                            valueExplorerTable.setModel(arrayValueTableModel);
                        }
                        else
                            throw new InternalError("Unknown NonNullReferenceValue: " + value.getClass().getName());

                        // todo
                        // this isn't working, it seems to give the cell bounds for the cell above
                        // have checked that getCellBounds takes a zero based index: adding 1 will
                        // throw an IndexOutOfBoundsException when invoked on the last index
                        SwingUtilities.invokeLater(new Runnable() {
                            public void run() {
                                heapTable.scrollRectToVisible(heapTable.getCellRect(index, 0, true));
                            }
                        });
                    }
                }
            });

            Heap heap = vm.getHeap();
            CallStack callStack = vm.getCallStack();

            callStackTableModel.setCallStack(callStack);
            heapTableModel.setHeap(heap);
            classTreeModel.setClassLoader(vm.getClassLoader());

            callStackTableSelectionModel.setSelectedIndex(0);
            heapTableSelectionModel.setSelectedIndex(0);

            heapInfoLabel.setText("Heap size " + heap.getSizeInBytes() + " bytes");

            heap.addHeapListener(new HeapListener() {
                public void newAllocation(HeapEvent e) {
                    heapInfoLabel.setText("Heap size " + ((Heap) e.getSource()).getSizeInBytes() + " bytes");
                }
            });

            vm.addPCListener(new PCListener() {
                public void pcChanged(PCEvent e) {
                    if (callStackTableSelectionModel.isSelectedIndex(0))
                        SwingUtilities.invokeLater(new Runnable() { public void run() {
                            try {
                                updateFileTextArea(vm.getCallStack().peek());
                            }
                            catch (NoSuchElementException e) {}
                        } } );
                    else
                        callStackTableSelectionModel.setSelectedIndex(0);
                }
            });

            vm.addVMListener(new VirtualMachineListener() {
                private void stopInterpreting() {
                    buttonPanelCardLayout.show(buttonPanel, "loadButtonPanel");
                }

                public void methodReturned(Value v) {
                    stopInterpreting();

                    StringBuffer message = new StringBuffer("Method returned");

                    if (v != null) {
                        message.append(" with value ");

                        if (v instanceof ReferenceValue) {
                            heapTableSelectionModel.setSelectedIndex(vm.getHeap().getIndexForLocation(((NonNullReferenceValue) v).getHeapLocation()));
                            message.append(" at heap location ");
                        }

                        message.append(v);
                    }

                    JOptionPane.showMessageDialog(null, message.toString(), "Method return", JOptionPane.INFORMATION_MESSAGE);
                }

                public void methodReturned(CarmelException e) {
                    stopInterpreting();

                    ClassValue exception = e.getException();
                    short heapLocation = exception.getHeapLocation();
                    heapTableSelectionModel.setSelectedIndex(vm.getHeap().getIndexForLocation(heapLocation));
                    String exceptionName = exception.getType().getName();

                    JOptionPane.showMessageDialog(null, "Method threw exception of type " + exceptionName + " (at heap location " + heapLocation + ")", exceptionName, JOptionPane.INFORMATION_MESSAGE);
                }

                public void vmException(VerificationException e) {
                    stopInterpreting();

                    e.printStackTrace();
                    JOptionPane.showMessageDialog(null, "Static verification error (caught at runtime): " + e.getMessage(), "Verification Error", JOptionPane.ERROR_MESSAGE);
                }

                public void vmException(Exception e) {
                    stopInterpreting();

                    e.printStackTrace();
                    JOptionPane.showMessageDialog(null, e.getMessage(), "Exception", JOptionPane.ERROR_MESSAGE);
                }

                public void vmHalted() {
                    stopInterpreting();
                }
            });

            setTableVisibleRowCount(callStackTable, 6);
            setTableVisibleRowCount(operandStackTable, 5);
            setTableVisibleRowCount(localVariablesTable, 5);
            setTableVisibleRowCount(heapTable, 6);
            setTableVisibleRowCount(valueExplorerTable, 6);
            setTableVisibleRowCount(staticFieldsTable, 6);

            // this is to make values right-aligned
            DefaultTableCellRenderer cellRenderer = new DefaultTableCellRenderer();
            cellRenderer.setHorizontalAlignment(JLabel.RIGHT);

            operandStackTable.setDefaultRenderer(StackEntry.class, cellRenderer);
            localVariablesTable.setDefaultRenderer(StackEntry.class, cellRenderer);
            valueExplorerTable.setDefaultRenderer(StackEntry.class, cellRenderer);
            staticFieldsTable.setDefaultRenderer(StackEntry.class, cellRenderer);

            setTableColumnWidths(callStackTable);
            setTableColumnWidths(operandStackTable);
            setTableColumnWidths(localVariablesTable);
            setTableColumnWidths(heapTable);
            setTableColumnWidths(valueExplorerTable);
            setTableColumnWidths(staticFieldsTable);

    //        callStackTable.getColumnModel().getColumn(2).setPreferredWidth(callStackTable.getDefaultRenderer(String.class).getTableCellRendererComponent(callStackTable, "handler", false, false, 0, 2).getPreferredSize().width);

            pack();

            Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            Dimension frameSize = getSize();

            if (frameSize.height > screenSize.height) frameSize.height = screenSize.height;
            if (frameSize.width > screenSize.width) frameSize.width = screenSize.width;

            setLocation((screenSize.width - frameSize.width) / 2, (screenSize.height - frameSize.height) / 2);
            if (frameSize.height > screenSize.height - 30) setSize(frameSize.width, screenSize.height - 30);

            if (callStack.getDepth() != 0) updateFileTextArea(callStack.peek());

            show();
    //        splitPane.resetToPreferredSizes();
    //        int top = splitPane.getTopComponent().getHeight();
    //        double weight = ((double) top) / (top + splitPane.getBottomComponent().getHeight());
    //        splitPane.setDividerLocation(weight);
    //        splitPane.setResizeWeight(weight);

            splitPane.setDividerLocation(0.5);
            callStackSplitPane.setDividerLocation(0.5);
            heapSplitPane.setDividerLocation(0.67);
            sourceSplitPane.setDividerLocation(0.3);
            staticFieldsSplitPane.setDividerLocation(0.67);

            classTree.getSelectionModel().setSelectionPath(new TreePath(classTreeModel.getPackageNode(vm.getClassLoader().loadPackage("java.lang")).getPath()));
        }
        catch (Exception e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(null, e.getMessage(), "Error loading the virtual machine", JOptionPane.ERROR_MESSAGE);
            System.exit(1);
        }
    }

    public static void main(String[] args) {
        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        }
        catch (Exception e) {}

        if (args.length == 0)
            new VMFrame();
        else
            new VMFrame(args[0]);
    }

    protected void closeWindow(VirtualMachine vm) {
        System.exit(1);
        vm.halt();
        hide();
        dispose();
    }

    protected void setTableVisibleRowCount(JTable table, int rowCount) {
        table.setPreferredScrollableViewportSize(new Dimension(table.getPreferredSize().width, rowCount * table.getRowHeight()));
    }

    protected void setTableColumnWidths(JTable table) {
        try {
            PrototypeValueTableModel model = (PrototypeValueTableModel) table.getModel();
            TableColumnModel columns = table.getColumnModel();
            TableCellRenderer headerRenderer = table.getTableHeader().getDefaultRenderer();

            for (int i = 0; i < model.getColumnCount(); i++) {
                TableColumn column = columns.getColumn(i);
                column.setPreferredWidth(
                    table.getDefaultRenderer(model.getColumnClass(i)).getTableCellRendererComponent(
                        table, model.getPrototypeValue(i), false, false, 0, 0
                    ).getPreferredSize().width
                );
                column.setMinWidth(
                    headerRenderer.getTableCellRendererComponent(
                        table, column.getHeaderValue(), false, false, 0, i
                    ).getMinimumSize().width
                );
            }
        }
        catch (ClassCastException e) {}
    }

    protected void updateFileTextArea(StackFrame frame) {
        carmelSourcePanel.setCarmelSource(frame.getCarmelSource());

        // Parser line numbers are 1-based
        try {
            carmelSourcePanel.highlightLine(frame.getPC().getLineNumber() - 1);
        }
        catch (BadLocationException e) {}
    }

    // overridden to fix bug in which inserting before the selected index
    // results in more than one selected index
    protected class SingleListSelectionModel extends DefaultListSelectionModel {
        SingleListSelectionModel() {
            setSelectionMode(SINGLE_SELECTION);
        }

        public int getSelectedIndex() {
            return getAnchorSelectionIndex();
        }

        public void setSelectedIndex(int index) {
            setSelectionInterval(index, index);
        }

        public void insertIndexInterval(int index, int length, boolean before) {
            super.insertIndexInterval(index, length, before);

            // select last index added
            index += length;
	    if (before) index--;

            setSelectionInterval(index, index);
        }

        public void removeIndexInterval(int index0, int index1) {
            boolean select0 = index0 == 0 && isSelectedIndex(0);

            super.removeIndexInterval(index0, index1);

            // if index 0 was selected, select new index 0
            if (select0) setSelectionInterval(0, 0);
        }
    }

    private void jbInit() throws Exception {
        heapBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Heap");
        staticFieldsBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Static Fields");
        callStackBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Call Stack");
        localVariablesBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Local Variables");
        operandStackBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Operand Stack");
        fileViewerBorder = new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Source Viewer");
        this.getContentPane().setLayout(borderLayout3);
        callStackSplitPane.setBorder(null);
        callStackSplitPane.setResizeWeight(0.5);
        operandStackScrollPane.setBorder(operandStackBorder);
        callStackTable.setModel(callStackTableModel);
        callStackTable.setSelectionModel(callStackTableSelectionModel);
        callStackTable.setPreferredScrollableViewportSize(new Dimension(300, 300));
        localVariablesTable.setModel(localVariablesTableModel);
        framePanel.setLayout(gridBagLayout3);
        operandStackTable.setModel(operandStackTableModel);
        vmStatePanel.setLayout(gridBagLayout2);
        localVariablesScrollPane.setBorder(localVariablesBorder);
        splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
        splitPane.setResizeWeight(0.5);
        callStackScrollPane.setBorder(callStackBorder);
        heapPanel.setLayout(borderLayout2);
        heapSplitPane.setBorder(heapBorder);
        heapSplitPane.setResizeWeight(0.5);
        heapTable.setModel(heapTableModel);
        heapTable.setSelectionModel(heapTableSelectionModel);
        carmelSourcePanel.setBorder(fileViewerBorder);
        carmelSourcePanel.setReadOnly(true);
        carmelSourcePanel.setRows(12);
        carmelSourcePanel.setColumns(50);
        staticFieldsSplitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
        staticFieldsSplitPane.setBorder(new TitledBorder(new EtchedBorder(EtchedBorder.RAISED,Color.white,new Color(165, 163, 151)),"Loaded Classes"));
        staticFieldsSplitPane.setResizeWeight(0.5);
        staticFieldsTable.setModel(staticFieldsTableModel);
        classTree.setModel(classTreeModel);
        classTree.setRootVisible(false);
        classTree.setShowsRootHandles(true);
        classTree.setVisibleRowCount(6);
        staticFieldsScrollPane.setBorder(staticFieldsBorder);
        stepIntoButton.setText("Step Into");
        closeButton.setText("Close");
        stepOutButton.setText("Step Out");
        stepOverButton.setText("Step Over");
        buttonPanel.setLayout(buttonPanelCardLayout);
        loadPackageButton.setText("Load Package...");
        loadFileButton.setText("Load File...");
        setClassPathButton.setText("Set Classpath...");
        runMethodButton.setText("Run Method");
        close2Button.setText("Close");
        haltButton.setText("Halt");
        valueExplorerTable.setModel(classValueTableModel);
        this.getContentPane().add(splitPane,  BorderLayout.CENTER);
        splitPane.add(vmStatePanel, JSplitPane.TOP);
        framePanel.add(localVariablesScrollPane, new GridBagConstraints(1, 0, 1, 1, 1.0, 1.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
        framePanel.add(operandStackScrollPane,   new GridBagConstraints(0, 0, 1, 1, 0.6, 1.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
        callStackSplitPane.add(callStackScrollPane, JSplitPane.LEFT);
        callStackScrollPane.getViewport().add(callStackTable, null);
        operandStackScrollPane.getViewport().add(operandStackTable, null);
        localVariablesScrollPane.getViewport().add(localVariablesTable, null);
        callStackSplitPane.add(framePanel, JSplitPane.RIGHT);
        vmStatePanel.add(callStackSplitPane,    new GridBagConstraints(0, 0, 1, 1, 1.0, 1.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
        splitPane.add(sourceSplitPane, JSplitPane.BOTTOM);
        sourceSplitPane.add(carmelSourcePanel, JSplitPane.RIGHT);
        sourceSplitPane.add(staticFieldsSplitPane, JSplitPane.LEFT);
        sourceSplitPane.setResizeWeight(0.7);
        staticFieldsSplitPane.add(classTreeScrollPane, JSplitPane.LEFT);
        staticFieldsSplitPane.add(staticFieldsScrollPane, JSplitPane.RIGHT);
        this.getContentPane().add(buttonPanel,  BorderLayout.SOUTH);
        buttonPanel.add(loadButtonPanel,  "loadButtonPanel");
        loadButtonPanel.add(loadPackageButton, null);
        loadButtonPanel.add(loadFileButton, null);
        loadButtonPanel.add(setClassPathButton, null);
        loadButtonPanel.add(runMethodButton, null);
        loadButtonPanel.add(close2Button, null);
        buttonPanel.add(runButtonPanel,   "runButtonPanel");
        runButtonPanel.add(stepIntoButton, null);
        runButtonPanel.add(stepOverButton, null);
        runButtonPanel.add(stepOutButton, null);
        runButtonPanel.add(haltButton, null);
        runButtonPanel.add(closeButton, null);
        staticFieldsSplitPane.setResizeWeight(0.33);
        staticFieldsScrollPane.getViewport().add(staticFieldsTable, null);
        classTreeScrollPane.getViewport().add(classTree, null);
        heapSplitPane.add(heapPanel, JSplitPane.LEFT);
        heapSplitPane.setResizeWeight(0.67);
        heapPanel.add(heapTableScrollPane, BorderLayout.CENTER);
        heapPanel.add(heapInfoLabel, BorderLayout.SOUTH);
        heapSplitPane.add(valueExplorerScrollPane, JSplitPane.RIGHT);
        valueExplorerScrollPane.getViewport().add(valueExplorerTable, null);
        heapTableScrollPane.getViewport().add(heapTable, null);
        vmStatePanel.add(heapSplitPane,   new GridBagConstraints(0, 1, 1, 1, 1.0, 1.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    }
}