package carmel.gui;

import carmel.tree.*;
import carmel.value.BottomValue;
import carmel.value.Value;
import java.util.*;
import javax.swing.table.AbstractTableModel;

public class StaticFieldTableModel extends AbstractTableModel implements StaticFieldListener, PrototypeValueTableModel {

    protected static final String[] columnNames = new String[] { "Name",       "Value",     "Type" };
    protected static final Class[] columnClasses = new Class[] { String.class, Value.class, String.class };
    protected static final Object[] prototypeValues = new Object[] { "fieldName", BottomValue.BOTTOM, "undef" };

    protected String[] fieldNames;
    protected TreeClassOrInterface classOrInterface;

    public StaticFieldTableModel() {}

    public StaticFieldTableModel(TreeClassOrInterface c) {
        setClassOrInterface(c);
    }

    public TreeClassOrInterface getClassOrInterface() { return classOrInterface; }

    public void setClassOrInterface(TreeClassOrInterface classOrInterface) {
        if (this.classOrInterface != null)
            for (Iterator i = this.classOrInterface.getDeclaredStaticFields().iterator(); i.hasNext();)
                ((TreeStaticField) i.next()).removeStaticFieldListener(this);

        this.classOrInterface = classOrInterface;

        if (classOrInterface != null) {
            Collection fields = classOrInterface.getDeclaredStaticFields();
            fieldNames = new String[fields.size()];
            int index = 0;

            for (Iterator i = fields.iterator(); i.hasNext();) {
                TreeStaticField field = (TreeStaticField) i.next();
                fieldNames[index++] = field.getMemberName();
                field.addStaticFieldListener(this);
            }

            Arrays.sort(fieldNames);
        }
        else
            fieldNames = null;

        fireTableDataChanged();
    }

    public void fieldChanged(StaticFieldEvent e) {
        int index = Arrays.binarySearch(fieldNames, ((TreeStaticField) e.getSource()).getName());

        if (index >= 0) fireTableRowsUpdated(index, index);
    }

    public int getRowCount() { return fieldNames == null ? 0 : fieldNames.length; }

    public int getColumnCount() { return 3; }

    public String getColumnName(int columnIndex) {
        return columnNames[columnIndex];
    }

    public Class getColumnClass(int columnIndex) {
        return columnClasses[columnIndex];
    }

    public Object getValueAt(int rowIndex, int columnIndex) {
        String fieldName = fieldNames[rowIndex];

        if (columnIndex == 0) return fieldName;

        try {
            TreeStaticField field = classOrInterface.getDeclaredStaticField(fieldName);

            switch (columnIndex) {
                case 1: return field.getValue();
                case 2: return field.getType().getJCVMType().getName();
                default: throw new IndexOutOfBoundsException();
            }
        }
        catch (NoSuchFieldException e) {
            return "NoSuchFieldException";
        }
    }

    public Object getPrototypeValue(int column) {
        return prototypeValues[column];
    }
}