package carmel.gui;

import carmel.interpreter.*;
import carmel.tree.*;
import java.util.*;
import javax.swing.tree.*;

public class ClassLoaderTreeModel extends DefaultTreeModel implements ClassLoaderListener {

    protected CarmelClassLoader classLoader;
    protected RootNode rootNode = new RootNode();

    public ClassLoaderTreeModel() {
        super(new DefaultMutableTreeNode());
        setRoot(rootNode);
    }

    public ClassLoaderTreeModel(CarmelClassLoader classLoader) {
        this();
        setClassLoader(classLoader);
    }

    public CarmelClassLoader getClassLoader() { return classLoader; }

    public synchronized void setClassLoader(CarmelClassLoader classLoader) {
        if (this.classLoader != null) this.classLoader.removeClassLoaderListener(this);

        this.classLoader = classLoader;

        if (classLoader != null) {
            classLoader.addClassLoaderListener(this);

//            int count = rootNode.getChildCount();
//            int[] indices = new int[count];
//            Object[] children = new Object[count];
//
//            for (int i = 0; i < count; i++) {
//                indices[i] = i;
//                children[i] = rootNode.getChildAt(i);
//            }
//
            rootNode.removeAllChildren();

//            fireTreeNodesRemoved(this, new Object[] { rootNode }, indices, children);

            for (Iterator i = classLoader.getPackages().iterator(); i.hasNext();)
                rootNode.addPackageNode(new PackageNode((TreePackage) i.next()));
        }

        // todo: this is a bit much, but then the other events don't seem to work
        reload();
    }

    public PackageNode getPackageNode(TreePackage p) {
        return rootNode.getPackageNode(p);
    }

    public synchronized void packageLoaded(ClassLoaderEvent event) {
        rootNode.addPackageNode(new PackageNode(event.getPackage()));
    }

    public class RootNode extends DefaultMutableTreeNode {

        public void addPackageNode(PackageNode node) {
            int index = children == null ? 0 : -Collections.binarySearch(children, node) - 1;

            if (index >= 0) {
                insert(node, index);
                nodesWereInserted(this, new int[] { index });
            }
        }

        public PackageNode getPackageNode(TreePackage p) {
            int index = children == null ? - 1 : Collections.binarySearch(children, new PackageNode(p));

            return (index < 0) ? null : (PackageNode) children.get(index);
        }
    }

    public interface Node {
        public void visit(ClassLoaderNodeVisitor v);
    }

    public class PackageNode extends DefaultMutableTreeNode implements Node, Comparable {
        protected TreePackage p;
        protected String name;

        protected PackageNode(TreePackage p) {
            super(p.getName() == null ? "<default>" : p.getName());

            this.p = p;
            name = p.getName();
            if (name == null) name = "<default>";

            TreeMap nodeMap = new TreeMap();

            for (Iterator i = p.interfaces.values().iterator(); i.hasNext();) {
                TreeInterface iface = (TreeInterface) i.next();
                nodeMap.put(iface.getClassName(), new InterfaceNode(iface));
            }

            for (Iterator i = p.classes.values().iterator(); i.hasNext();) {
                TreeClass c = (TreeClass) i.next();
                nodeMap.put(c.getClassName(), new ClassNode(c));
            }

            for (Iterator i = nodeMap.values().iterator(); i.hasNext();)
                add((ClassNode) i.next());
        }

        public TreePackage getPackage() {
            return p;
        }

        public int hashCode() { return p.name.hashCode(); }

        public boolean equals(Object o) {
            return o instanceof PackageNode && name.equals(((PackageNode) o).name);
        }

        public int compareTo(Object o) {
            return name.compareTo(((PackageNode) o).name);
        }

        public void visit(ClassLoaderNodeVisitor v) {
            v.visit(this);
        }
    }

    public abstract class ClassOrInterfaceNode extends DefaultMutableTreeNode implements Node {
        protected TreeClassOrInterface classOrInterface;

        protected ClassOrInterfaceNode(TreeClassOrInterface classOrInterface) {
            super(classOrInterface.getClassName());

            this.classOrInterface = classOrInterface;
        }

        public void addMethods() {
            TreeMap nodeMap = new TreeMap();

            for (Iterator i = classOrInterface.getDeclaredMethods().iterator(); i.hasNext();) {
                TreeMethod m = (TreeMethod) i.next();
                nodeMap.put(m.getMemberNameWithTypes(), new MethodNode(m));
            }

            for (Iterator i = nodeMap.values().iterator(); i.hasNext();)
                add((MethodNode) i.next());
        }

        public TreeClassOrInterface getClassOrInterface() {
            return classOrInterface;
        }
    }

    public class ClassNode extends ClassOrInterfaceNode {
        protected ClassNode(TreeClass c) {
            super(c);

            TreeMap nodeMap = new TreeMap();

            for (Iterator i = c.getConstructors().iterator(); i.hasNext();) {
                TreeConstructor m = (TreeConstructor) i.next();
                nodeMap.put(m.getMemberNameWithTypes(), new ConstructorNode(m));
            }

            for (Iterator i = nodeMap.values().iterator(); i.hasNext();)
                add((ConstructorNode) i.next());

            addMethods();
        }

        public TreeClass getTreeClass() {
            return (TreeClass) classOrInterface;
        }

        public void visit(ClassLoaderNodeVisitor v) {
            v.visit(this);
        }
    }

    public class InterfaceNode extends ClassOrInterfaceNode {
        protected InterfaceNode(TreeInterface i) {
            super(i);
            addMethods();
        }

        public TreeInterface getInterface() {
            return (TreeInterface) classOrInterface;
        }

        public void visit(ClassLoaderNodeVisitor v) {
            v.visit(this);
        }
    }

    public abstract class ConstructorOrMethodNode extends DefaultMutableTreeNode implements Node {
        protected TreeConstructorOrMethod method;

        protected ConstructorOrMethodNode(TreeConstructorOrMethod method) {
            super(method.getMemberNameWithTypes());
            this.method = method;
        }

        public TreeConstructorOrMethod getConstructorOrMethod() {
            return method;
        }
    }

    public class ConstructorNode extends ConstructorOrMethodNode {
        protected ConstructorNode(TreeConstructor c) {
            super(c);
        }

        public TreeConstructor getConstructor() {
            return (TreeConstructor) method;
        }

        public void visit(ClassLoaderNodeVisitor v) {
            v.visit(this);
        }
    }

    public class MethodNode extends ConstructorOrMethodNode {
        protected MethodNode(TreeMethod m) {
            super(m);
        }

        public TreeMethod getMethod() {
            return (TreeMethod) method;
        }

        public void visit(ClassLoaderNodeVisitor v) {
            v.visit(this);
        }
    }
}