package carmel.gui;

import carmel.interpreter.CarmelSource;
import java.awt.*;
import java.io.IOException;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

public class CarmelSourcePanel extends JPanel {

    protected JTextArea textArea = new JTextArea();
    protected JLabel label = new JLabel();

    protected Highlighter highlighter;
    protected Object highlightTag;
    protected CaretListener caretListener;

    protected CarmelSource source;
    protected boolean readOnly;
    protected boolean dirty;

    public CarmelSourcePanel() {
        this(false);
    }

    public CarmelSourcePanel(boolean readOnly) {
        setLayout(new BorderLayout());

        highlighter = textArea.getHighlighter();
        try {
            highlightTag = highlighter.addHighlight(0, 0, new DefaultHighlighter.DefaultHighlightPainter(Color.yellow));
        }
        catch (BadLocationException e) {
            throw new InternalError();
        }

        caretListener = new CaretListener() {
            public void caretUpdate(CaretEvent e) {
                if (source != null) updateLabelText();
            }
        };

        textArea.getDocument().addDocumentListener(new DocumentListener() {
            public void insertUpdate(DocumentEvent e) { dirty = true; }
            public void removeUpdate(DocumentEvent e) { dirty = true; }
            public void changedUpdate(DocumentEvent e) {}
        });

        add(new JScrollPane(textArea), BorderLayout.CENTER);
        add(label, BorderLayout.SOUTH);

        setReadOnly(readOnly);
    }

    public boolean isCurrentlyReadOnly() {
        return readOnly || source == null || source.isReadOnly();
    }

    public boolean isReadOnly() { return readOnly; }

    public void setReadOnly(boolean readOnly) {
        this.readOnly = readOnly;
        updateCurrentlyReadOnly();
    }

    protected void updateCurrentlyReadOnly() {
        boolean currentlyReadOnly = isCurrentlyReadOnly();

        textArea.setEditable(!currentlyReadOnly);
        if (currentlyReadOnly)
            textArea.removeCaretListener(caretListener);
        else
            textArea.addCaretListener(caretListener);

        updateLabelText();
    }

    protected void updateLabelText() {
        if (source == null) {
            label.setText("No source loaded");
        }
        else {
            int position = textArea.getCaretPosition();

            try {
                int line = textArea.getLineOfOffset(position);
                int column = position - textArea.getLineStartOffset(line);

                boolean currentlyReadOnly = isCurrentlyReadOnly();

                label.setText(source.getName() + ": line " + (line + 1) + ( currentlyReadOnly ? "" : ", column " + (column + 1) ) + ((!readOnly && currentlyReadOnly) ? " (read only)" : ""));
            }
            catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

    public CarmelSource getCarmelSource() { return source; }

    public void setCarmelSource(CarmelSource source) {
        if (source == null ? this.source != null : !source.equals(this.source)) {
            if (source == null) {
                this.source = null;
                textArea.setText("");
                updateLabelText();
            }
            else {
                try {
                    this.source = source;
                    textArea.setText(source.getContents());
                    updateLabelText();
                }
                catch (IOException e) {
                    this.source = null;
                    textArea.setText("");
                    label.setText("Error reading source " + source.getName() + ": " + e.getMessage());
                }
            }

            dirty = false;
            updateCurrentlyReadOnly();
        }

        // todo: turn off highlighter
    }

    public int getRows() {
        return textArea.getRows();
    }

    public void setRows(int rows) {
        textArea.setRows(rows);
    }

    public int getColumns() {
        return textArea.getColumns();
    }

    public void setColumns(int columns) {
        textArea.setColumns(columns);
    }

    public void save() throws IOException {
        if (!dirty || source == null || source.isReadOnly()) return;

        source.setContents(textArea.getText());

        dirty = false;
    }

    public void highlightLine(int line) throws BadLocationException {
        highlighter.changeHighlight(highlightTag, textArea.getLineStartOffset(line), textArea.getLineEndOffset(line));
        setCaretPosition(line, 0);
    }

    public void setCaretPosition(int line, int column) throws BadLocationException {
        int y = textArea.modelToView(textArea.getLineStartOffset(Math.max(line - 2, 0))).y;
        int height = textArea.modelToView(textArea.getLineStartOffset(Math.min(line + 2, textArea.getLineCount() - 1))).y + getFontMetrics(textArea.getFont()).getHeight() - y;

        textArea.setCaretPosition(textArea.getLineStartOffset(line) + column);
        textArea.scrollRectToVisible(new Rectangle(0, y, 0, height));
        textArea.requestFocus();

        if (isCurrentlyReadOnly()) updateLabelText();
    }
}